<?php

namespace App\Http\Controllers;

use App\Http\Requests\AcademicLevel\DeleteAcademicLevelRequest;
use App\Http\Requests\AcademicLevel\StoreAcademicLevelRequest;
use App\Http\Requests\AcademicLevel\UpdateAcademicLevelRequest;
use App\Http\Requests\AcademicLevel\ViewAcademicLevelRequest;
use App\Models\AcademicLevel;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use DB;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class AcademicLevelController extends Controller
{

    public function index(ViewAcademicLevelRequest $request)
    {
        try {
            $dataTableTools = new DataTableTools();
            $query = AcademicLevel::select();

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('academic_levels', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'پایه'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreAcademicLevelRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $academicLevelAdd = CommonTools::safeRequest($request, 'academic_level');


            DB::beginTransaction();

            $academicLevel = AcademicLevel::create($academicLevelAdd);


            $user->appendLog($request, ['new' => $academicLevel]);
            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'پایه'])
                ->setData('academic_level', $academicLevel)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'academicLevelStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'پایه'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteAcademicLevelRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $academicLevel = AcademicLevel::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $academicLevel->toArray()]);
            $academicLevel->delete();


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'پایه'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'academicLevelDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'پایه'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateAcademicLevelRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $user = User::find(auth()->user()->id ?? -1);
            $requestBody = CommonTools::safeRequest($request, 'id', 'academic_level');
            $academicLevelUpdate = $requestBody['academic_level'];
            $academicLevelOriginal = AcademicLevel::find($requestBody['id']);

            DB::beginTransaction();

            $user->appendLog($request, [
                'old' => $academicLevelOriginal->toArray(),
                'new' => $academicLevelUpdate,
            ]);
            $academicLevelOriginal->update($academicLevelUpdate);

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_updated', ['title' => 'پایه'])
                ->setData('academic_level', $academicLevelOriginal)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'academicLevelUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'پایه'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
