<?php

namespace App\Http\Controllers;

use App\Enums\ApplicationConfigActions;
use App\Enums\PermissionApplicationConfigAccess;
use App\Http\Requests\ApplicationConfig\UpdateApplicationConfigSchoolInformationRequest;
use App\Http\Requests\ApplicationConfig\UpdateApplicationConfigTermRequest;
use App\Http\Requests\ApplicationConfig\ViewApplicationConfigRequest;
use App\Models\ApplicationConfig;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\FileTools;
use App\Tools\ResponseTools;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class ApplicationConfigController extends Controller
{
    public function index(ViewApplicationConfigRequest $request)
    {
        try {
            $user = User::find(auth()->user()->id ?? -1);
            $isNotAdmin = true;

            if ($user) {
                $isNotAdmin = !$user->hasPermissions(
                    [
                        PermissionApplicationConfigAccess::view
                    ]
                );
            }

            $dataTableTools = new DataTableTools();
            $query = ApplicationConfig::select()
                ->when($isNotAdmin, function ($q) {
                    return $q->where('is_private', '=', false)
                        ->where('active', '=', true);
                });

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('application_configs', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'تنظیمات برنامه']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function updateSchoolInformation(
        UpdateApplicationConfigSchoolInformationRequest $request
    ) {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);

            if ($user->getAccess() !== User::ACCESS_ADMINISTRATOR) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessageFormat('messages.access_denied_to_page')
                    ->getJsonResponse(Response::HTTP_UNAUTHORIZED);
            }

            $applicationConfigUpdate = CommonTools::safeRequest(
                $request,
                'school'
            );


            if (isset($applicationConfigUpdate['image'])) {
                (new FileTools())
                    ->uploadFile(
                        FileTools::$STORE_SCHOOL,
                        $applicationConfigUpdate['image'],
                        'logo'
                    )
                    ->store();
            }

            $schoolInformationOriginal = ApplicationConfig::whereEncrypted(
                'title',
                '=',
                ApplicationConfigActions::schoolControllerInformationUpdate
            )
                ->firstOrFail();

            $schoolPrivateInformationOriginal = ApplicationConfig::whereEncrypted(
                'title',
                '=',
                ApplicationConfigActions::schoolControllerPrivateInformationUpdate
            )
                ->firstOrFail();


            DB::beginTransaction();

            $user->appendLog($request, [
                'old' => [
                    'information' => $schoolInformationOriginal->toArray(),
                    'privateInformation' => $schoolPrivateInformationOriginal->toArray()
                ],
                'new' => $applicationConfigUpdate,
            ]);

            $schoolInformationOriginal->update([
                'value' => [
                    'title' => $applicationConfigUpdate['title'],
                    'province' => $applicationConfigUpdate['province'],
                    'district' => $applicationConfigUpdate['district'],
                ]
            ]);

            $schoolPrivateInformationOriginal->update([
                'value' => [
                    'school_code' => $applicationConfigUpdate['code'],
                    'school_type' => $applicationConfigUpdate['type'],
                ]
            ]);

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'تنظیمات برنامه']
                )
                ->setData('application_configs', [
                    $schoolInformationOriginal->toArray(),
                    $schoolPrivateInformationOriginal->toArray(),
                ])
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'applicationConfigUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_updating_information',
                ['title' => 'تنظیمات برنامه']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function updateTerm(
        UpdateApplicationConfigTermRequest $request
    ) {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);

            $applicationConfigUpdate = CommonTools::safeRequest(
                $request,
                'terms'
            );

            $schoolPrivateTermOriginal = ApplicationConfig::whereEncrypted(
                'title',
                '=',
                ApplicationConfigActions::schoolControllerPrivateTermUpdate
            )
                ->firstOrFail();

            DB::beginTransaction();

            $user->appendLog($request, [
                'old' => [
                    'privateInformation' => $schoolPrivateTermOriginal->toArray()
                ],
                'new' => $applicationConfigUpdate,
            ]);

            $schoolPrivateTermOriginal->update([
                'value' => $applicationConfigUpdate
            ]);

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'تنظیمات برنامه']
                )
                ->setData('application_configs', [
                    $schoolPrivateTermOriginal->toArray(),
                ])
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'applicationConfigUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_updating_information',
                ['title' => 'تنظیمات برنامه']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
