<?php

namespace App\Http\Controllers;

use App\Http\Requests\Auth\RefreshAuthRequest;
use App\Http\Requests\Auth\SignInAuthRequest;
use App\Models\User;
use App\Http\Requests\Auth\SignUpAuthRequest;
use App\Models\Role;
use App\Tools\CommonTools;
use App\Tools\RandomGenerator;
use App\Tools\Regex;
use App\Tools\ResponseTools;
use Carbon\Carbon;
use DB;
use ESolution\DBEncryption\Encrypter;
use Exception;
use Illuminate\Http\JsonResponse;
use Symfony\Component\HttpFoundation\Response;

class AuthController extends Controller
{

    private const TTL = 20;
    private const TTL_LONG = 43200;
    private const ERROR_CODE_ACTIVE = 1;
    private const ERROR_CODE_VERIFY_AUTH = 2;

    public function signIn(SignInAuthRequest $request): JsonResponse
    {
        try {
            if ($request->fails()) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessage($request->firstError())
                    ->getJsonResponse(Response::HTTP_BAD_REQUEST);
            }

            $user = null;
            $token = null;
            $name = $request->input('name');
            $isPhone = preg_match(Regex::usernamePhoneNumber, $name);

            $isTTLLong = $request->get('app', false);
            $ttl = boolval($isTTLLong) == true ? self::TTL_LONG : self::TTL;
            $credentials = ['name' => Encrypter::encrypt($name), 'password' => $request->input('password')];

            if ($isPhone) {
                $user = User::whereHas('profile', function ($qProfile) use ($name) {
                    return $qProfile->whereEncrypted('phone', $name);
                })
                    ->first();

                if (isset($user)) {
                    if (! password_verify($request->input('password'), $user->password)) {
                        return ResponseTools::getInstance()
                            ->setStatus(ResponseTools::RES_ERROR)
                            ->setMessageFormat('messages.wrong_username_password')
                            ->getJsonResponse(Response::HTTP_BAD_REQUEST);
                    }

                    $token = auth()->setTTL($ttl)->login($user);
                }
            } else {
                $token = auth()->setTTL($ttl)->attempt($credentials);
            }

            if (
                isset($token) && strlen($token) > 10
            ) {
                $user = User::find(auth()->user()->id ?? -1);

                if ($user->id != 1) {
                    if ($user->active != 1 && $user->active != 2) {
                        return ResponseTools::getInstance()
                            ->setStatus(ResponseTools::RES_ERROR)
                            ->setData('error_code', self::ERROR_CODE_ACTIVE)
                            ->setMessage(__('messages.access_denied_active'))
                            ->getJsonResponse(Response::HTTP_UNAUTHORIZED);
                    }
                }

                $role = $user->roles()->whereIn('primary', [1, 2])->firstOrFail();

                $roleName = $role->name;
                $roleTitle = $role->title;
                $returnedPermissions = $role->permissions->pluck("name");

                return ResponseTools::getInstance()
                    ->setMessage(__('messages.welcome_to_site'))
                    ->setData('user', [
                        'user_id' => $user->id,
                        'token' => $token,
                        'expireTokenTime' => $ttl,
                        'role' => $roleName,
                        'role_title' => $roleTitle,
                        'permissions' => $returnedPermissions,
                    ])
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage(__('messages.wrong_username_password'))
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Exception $exception) {
            auth()->check() && auth('api')->logout();

            CommonTools::registerException($exception, 'user-login');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessage(__('messages.an_error_has_occurred_contact_the_system_administrator'))
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function signUp(SignUpAuthRequest $request): JsonResponse
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $userAdd = CommonTools::safeRequest($request, 'user');

            $password = $userAdd['password'] ?? RandomGenerator::password();
            $username = $userAdd['username'] ?? (RandomGenerator::password(5) . Carbon::now()->unix());
            $personRoleId = Role::select("id")
                ->where("name", "personalAccount")
                ->firstOrFail()
                ->id;

            $isTTLLong = $request->get('app', false);
            $ttl = boolval($isTTLLong) == true ? self::TTL_LONG : self::TTL;

            DB::beginTransaction();

            $user = User::create(
                [
                    'name' => $username,
                    'password' => $password,
                ]
            );

            $user->profile()
                ->create($userAdd['profile']);

            $user->roles()
                ->attach(
                    $personRoleId,
                    [
                        'primary' => 2,
                    ]
                );

            DB::commit();

            $token = auth()
                ->setTTL($ttl)
                ->login($user);

            $role = $user->roles()->whereIn('primary', [1, 2])->firstOrFail();

            $roleName = $role->name;
            $roleTitle = $role->title;
            $returnedPermissions = $role->permissions->pluck("name");

            return ResponseTools::getInstance()
                ->setMessage(__('messages.welcome_to_site'))
                ->setData('user', [
                    'user_id' => $user->id,
                    'token' => $token,
                    'expireTokenTime' => $ttl,
                    'role' => $roleName,
                    'role_title' => $roleTitle,
                    'permissions' => $returnedPermissions,
                ])
                ->getJsonResponse();
        } catch (Exception $e) {
            CommonTools::registerException($e, 'signUp');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessage(__('messages.an_error_has_occurred_contact_the_system_administrator'))
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function signOut(): JsonResponse
    {
        if ($user = auth('api')->user()) {
            try {
                auth('api')->logout();

                return ResponseTools::getInstance()
                    ->setMessage(__('messages.logout_completed_successfully'))
                    ->getJsonResponse();
            } catch (Exception $exception) {
                CommonTools::registerException($exception, 'user-logout');
            }

            return ResponseTools::getInstance()
                ->setMessage(__('messages.an_error_has_occurred_contact_the_system_administrator'))
                ->setStatus(ResponseTools::RES_ERROR)
                ->getJsonResponse(Response::HTTP_UNAUTHORIZED);
        }

        return ResponseTools::getInstance()
            ->setMessage(__('messages.logout_completed_successfully'))
            ->getJsonResponse();
    }

    public function refresh(RefreshAuthRequest $request): JsonResponse
    {
        try {
            $isTTLLong = $request->get('app', false);
            $ttl = boolval($isTTLLong) == true ? self::TTL_LONG : self::TTL;
            $token = auth('api')->getToken();
            $new_token = auth('api')->setTTL($ttl)->refresh($token);

            return ResponseTools::getInstance()
                ->setMessage(__('messages.refresh_token_was_performed'))
                ->setData('user', [
                    'token' => $new_token,
                    'expireTokenTime' => $ttl,
                ])
                ->getJsonResponse();
        } catch (Exception $exception) {
        }

        return ResponseTools::getInstance()
            ->setMessage(__('messages.an_error_has_occurred_contact_the_system_administrator'))
            ->setStatus(ResponseTools::RES_ERROR)
            ->getJsonResponse(Response::HTTP_UNAUTHORIZED);
    }
}
