<?php

namespace App\Http\Controllers;

use App\Enums\PermissionClassRoomAccess;
use App\Http\Requests\ClassRoom\DeleteClassRoomRequest;
use App\Http\Requests\ClassRoom\StoreClassRoomRequest;
use App\Http\Requests\ClassRoom\UpdateClassRoomRequest;
use App\Http\Requests\ClassRoom\ViewClassRoomRequest;
use App\Http\Requests\ClassRoom\ViewStudentsClassRoomRequest;
use App\Models\ClassRoom;
use App\Models\Student;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use DB;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class ClassRoomController extends Controller
{

    public function index(ViewClassRoomRequest $request)
    {
        try {
            $user = User::find(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions([PermissionClassRoomAccess::view]);
            $dataTableTools = new DataTableTools();

            $query = ClassRoom::with([
                'major',
                'academicLevel',
            ])
                ->when(
                    $isNotAdmin,
                    fn($qClassRoom) =>
                    $qClassRoom->whereHas(
                        'weeklySchedules',
                        fn($qWeeklySchedule) =>
                        $qWeeklySchedule->where(
                            'teacher_id',
                            '=',
                            $user->id
                        )
                    )
                );

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('class_rooms', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'کلاس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function indexStudents(ViewStudentsClassRoomRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::findOrFail(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions(PermissionClassRoomAccess::view);
            $classRoomId = CommonTools::safeRequest($request, 'id');


            $dataTableTools = new DataTableTools();
            $query = Student::with([])
                ->whereHas(
                    'classRooms',
                    fn($qClassRoom) =>
                    $qClassRoom
                        ->where('class_rooms.id', '=', $classRoomId)
                        ->when(
                            $isNotAdmin,
                            fn($wqClassRoom) =>
                            $wqClassRoom->whereHas(
                                'weeklySchedules',
                                fn($qWeeklySchedule) =>
                                $qWeeklySchedule->where(
                                    'teacher_id',
                                    '=',
                                    $user->id
                                )
                            )
                        )
                );

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('students', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'کلاس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreClassRoomRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $classRoomAdd = CommonTools::safeRequest($request, 'class_room');


            DB::beginTransaction();

            $classRoom = ClassRoom::create($classRoomAdd);


            $user->appendLog($request, ['new' => $classRoom]);
            DB::commit();

            $classRoom->load([
                'major',
                'academicLevel',
            ]);

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'کلاس'])
                ->setData('class_room', $classRoom)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'classRoomStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'کلاس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteClassRoomRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $classRoom = ClassRoom::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $classRoom->toArray()]);
            $classRoom->delete();


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'کلاس'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'classRoomDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'کلاس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateClassRoomRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $user = User::find(auth()->user()->id ?? -1);
            $requestBody = CommonTools::safeRequest($request, 'id', 'class_room');
            $classRoomUpdate = $requestBody['class_room'];
            $classRoomOriginal = ClassRoom::find($requestBody['id']);

            DB::beginTransaction();

            $user->appendLog($request, [
                'old' => $classRoomOriginal->toArray(),
                'new' => $classRoomUpdate,
            ]);
            $classRoomOriginal->update($classRoomUpdate);

            DB::commit();

            $classRoomOriginal->load([
                'major',
                'academicLevel',
            ]);

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_updated', ['title' => 'کلاس'])
                ->setData('class_room', $classRoomOriginal)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'classRoomUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'کلاس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
