<?php

namespace App\Http\Controllers;

use App\Enums\ApplicationConfigActions;
use App\Http\Requests\DailyReport\ExportPdfDailyReportRequest;
use App\Models\ApplicationConfig;
use App\Models\Bell;
use App\Models\Moral;
use App\Models\Presence;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use Arr;
use Carbon\Carbon;
use PDF;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class DailyReportController extends Controller
{

    public function exportPDF(ExportPdfDailyReportRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $date = CommonTools::safeRequest($request, 'date');

            $date = $date != null ? Carbon::createFromFormat('Y-m-d', $date) : Carbon::now();

            $applicationConfigs = ApplicationConfig::where(
                fn($qAppConfig) => $qAppConfig->whereEncrypted(
                    'title',
                    '=',
                    ApplicationConfigActions::schoolControllerInformationUpdate
                )
                    ->orWhereEncrypted(
                        'title',
                        '=',
                        ApplicationConfigActions::schoolControllerPrivateInformationUpdate
                    )
            )
                ->where(
                    'active',
                    '=',
                    true
                )
                ->limit(2)
                ->orderBy('title')
                ->get()
                ->toArray();

            [$schoolConfig, $schoolPrivateConfig] = $applicationConfigs;

            $presenceDataTableTools = new DataTableTools();
            $moralDataTableTools = new DataTableTools();

            $presenceQuery = Presence::select(
                [
                    'id',
                    'school_session_id',
                    'student_id',
                    'entry_time',
                    'exit_time',
                    'created_at'
                ]
            )
                ->with([
                    'student:id,first_name,last_name',
                    'bell:bells.id,started_at',
                ])
                ->where(
                    fn($qPresences) =>
                    $qPresences->whereNotNull('delay_reason')
                        ->orWhereNotNull('early_exit_reason')
                        ->orWhereNotNull('entry_time')
                        ->orWhereNotNull('exit_time')
                )
                ->whereDate('created_at',  $date->startOfDay());

            $moralQuery = Moral::select(
                [
                    'id',
                    'school_session_id',
                    'student_id',
                    'moral',
                    'comment',
                    'created_at'
                ]
            )
                ->with([
                    'student:id,first_name,last_name',
                    'bell:bells.id',
                ])
                ->where('moral', '<', 2)
                ->whereDate('created_at',  $date->startOfDay());;

            $presenceDataTableTools->doQuery($presenceQuery, $request);

            $moralDataTableTools->doQuery($moralQuery, $request);

            if (
                $presenceDataTableTools->status == ResponseTools::RES_SUCCESS ||
                $moralDataTableTools->status == ResponseTools::RES_SUCCESS
            ) {
                $presenceQuery = $presenceDataTableTools->data
                    ->sortBy([
                        fn($a, $b) => $this->sortByBell($a, $b),
                        fn($a, $b) => $this->sortByLastName($a, $b),
                        fn($a, $b) => $this->sortByFirstName($a, $b),
                    ])
                    ->values()
                    ->groupBy('bell.id');

                $moralsQuery = $moralDataTableTools->data
                    ->sortBy([
                        fn($a, $b) => $this->sortByBell($a, $b),
                        fn($a, $b) => $this->sortByLastName($a, $b),
                        fn($a, $b) => $this->sortByFirstName($a, $b),
                    ])
                    ->values()
                    ->groupBy('bell.id');

                $presencesData = $presenceQuery
                    ->toArray();

                $moralsData = $moralsQuery
                    ->toArray();

                $bellIds = (clone($presenceQuery->keys()))
                    ->combine((clone($presenceQuery->keys())))
                    ->unique();

                $bells = Bell::whereIn('id', $bellIds)
                    ->get();

                $pdf = PDF::loadView('pdf.daily-report-presences-and-morals', [
                    'presences' => $presencesData,
                    'morals' => $moralsData,
                    'bells' => $bells,
                    'date' => $date,
                    'reportInfo' => [
                        'school_code' => $schoolPrivateConfig['value']['school_code'],
                        'school_type' => $schoolPrivateConfig['value']['school_type'],
                        'school_title' => $schoolConfig['value']['title'],
                        'school_province' => $schoolConfig['value']['province'],
                        'school_district' => $schoolConfig['value']['district'],
                    ]
                ]);

                return $pdf->stream('document.pdf');
            }

            return ResponseTools::getInstance()
                ->setStatus($moralDataTableTools->status)
                ->setMessage($moralDataTableTools->message)
                ->setData('presence_status', $presenceDataTableTools->status)
                ->setData('presence_message', $presenceDataTableTools->message)
                ->getJsonResponse();
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'حضور و غیاب']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    private function sortByBell($a,  $b)
    {
        $aFormat = Carbon::createFromFormat(
            'H:m:s',
            $a->bell->started_at ?? '00:00:00'
        )
            ->unix();

        $bFormat = Carbon::createFromFormat(
            'H:m:s',
            $b->bell->started_at ?? '00:00:00'
        )
            ->unix();

        return $aFormat <=> $bFormat;
    }

    private function sortByLastName($a,  $b)
    {
        return $a->student->last_name <=> $b->student->last_name;
    }

    private function sortByFirstName($a,  $b)
    {
        return $a->student->first_name <=> $b->student->first_name;
    }
}
