<?php

namespace App\Http\Controllers;

use App\Enums\ApplicationConfigActions;
use App\Enums\PermissionGradeAccess;
use App\Http\Requests\Grade\DeleteGradeRequest;
use App\Http\Requests\Grade\ExportPDFGradeRequest;
use App\Http\Requests\Grade\StoreGradeRequest;
use App\Http\Requests\Grade\UpdateGradeRequest;
use App\Http\Requests\Grade\ViewGradeRequest;
use App\Models\ApplicationConfig;
use App\Models\Grade;
use App\Models\SchoolSession;
use App\Models\Student;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use Arr;
use DB;
use PDF;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class GradeController extends Controller
{

    public function index(ViewGradeRequest $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions(PermissionGradeAccess::view);

            $dataTableTools = new DataTableTools();

            $query = Grade::with([
                'student:id,first_name,last_name',
                'schoolSession:id,weekly_schedule_id,teacher_user_id,held,created_at,updated_at',
                'schoolSession.teacher:id',
                'schoolSession.teacher.profile:id,user_id,first_name,last_name'
            ])
                ->when($isNotAdmin, fn($qGrade) =>
                $qGrade->whereHas(
                    'schoolSession',
                    fn($qSchoolSession) =>
                    $qSchoolSession->where('teacher_user_id', '=', $user->id)
                ));

            $dataTableTools->doQuery($query, $request)
                ->appendFields(['student.current_class_room_id'])
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('grades', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'نمره'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function exportPDF(ExportPDFGradeRequest $request)
    {

        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::findOrFail(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions(PermissionGradeAccess::exportPDF);

            $dataTableTools = new DataTableTools();

            $applicationConfigs = ApplicationConfig::where(
                fn($qAppConfig) => $qAppConfig->whereEncrypted(
                    'title',
                    '=',
                    ApplicationConfigActions::schoolControllerInformationUpdate
                )
                    ->orWhereEncrypted(
                        'title',
                        '=',
                        ApplicationConfigActions::schoolControllerPrivateInformationUpdate
                    )
            )
                ->where(
                    'active',
                    '=',
                    true
                )
                ->orderBy('title')
                ->get()
                ->toArray();

            [$schoolConfig, $schoolPrivateConfig] = $applicationConfigs;

            $query = Grade::select([
                'id',
                'school_session_id',
                'student_id',
                'lesson_id',
                'grade',
                'created_at',
            ])
                ->with([
                    'lesson:id,title,passing_grade',
                    'teacher:users.id',
                    'teacher.profile:profiles.id,user_id,first_name,last_name',
                ])
                ->when(
                    $isNotAdmin,
                    fn($qGrade) =>
                    $qGrade->whereHas(
                        'student',
                        fn($qStudent) =>
                        $qStudent->where(
                            'parent_id',
                            '=',
                            $user->id
                        )
                    )
                );
            $queryResult = $query->get();

            $data = $queryResult
                ->groupBy('student_id')
                ->toArray();

            $studentsKeyed = array_keys($data);
            $schoolSessionsKeyed = $queryResult
                ->pluck('school_session_id')
                ->unique();

            $students = Student::select(
                [
                    'id',
                    'parent_id',
                    'first_name',
                    'last_name',
                    'national_code',
                ]
            )
                ->with(['image', 'studentParent:id', 'studentParent.profile:id,user_id,first_name'])
                ->whereIn('id', $studentsKeyed)
                ->get();

            $schoolSessions = SchoolSession::select(
                [
                    'id',
                    'weekly_schedule_id',
                ]
            )
                ->with([
                    'classRoom:class_rooms.id,academic_level_id,major_id,code',
                    'classRoom.academicLevel:academic_levels.id,title',
                    'classRoom.major:majors.id,title',
                ])
                ->whereIn('id', $schoolSessionsKeyed)
                ->get();



            $studentGrades = $students
                ->map(function ($studentItem) use ($data, $schoolSessions) {
                    $schoolSession = null;
                    $grades = $data[$studentItem->id];

                    if (count($grades) > 0) {

                        $schoolSession = $schoolSessions->find($grades[0]['school_session_id']);

                        if ($schoolSession != null) {
                            $schoolSession = $schoolSession->toArray();
                        }
                    }

                    return array_merge(
                        $studentItem->toArray(),
                        [
                            'grades' => $grades,
                            'school_session' => $schoolSession,
                        ]
                    );
                });

            $dataTableTools->doQuery($query, $request);
            $gradeQueryWrapFilters = (new DataTableTools())->wrapFilters($query, $request);
            $firstGrade = (clone $gradeQueryWrapFilters)->orderBy('created_at')->first();
            $lastGrade = (clone $gradeQueryWrapFilters)->orderByDesc('created_at')->first();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                $pdf = PDF::loadView('pdf.student-grades', [
                    'studentGrades' => $studentGrades,
                    'firstGrade' => $firstGrade,
                    'lastGrade' => $lastGrade,
                    'reportInfo' => [
                        'school_code' => $schoolPrivateConfig['value']['school_code'],
                        'school_type' => $schoolPrivateConfig['value']['school_type'],
                        'school_title' => $schoolConfig['value']['title'],
                        'school_province' => $schoolConfig['value']['province'],
                        'school_district' => $schoolConfig['value']['district'],
                    ]
                ]);

                return $pdf->stream('document.pdf');
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            CommonTools::registerException($e, 'gradeExportPDF');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'حضور و غیاب']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreGradeRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $gradesAdd = CommonTools::safeRequest($request, 'grades');
            $grades = [];

            $schoolSessions = SchoolSession::select(
                [
                    'id',
                    'weekly_schedule_id',
                ]
            )
                ->with(
                    [
                        'weeklySchedule:weekly_schedules.id,lesson_id'
                    ]
                )
                ->whereIn(
                    'id',
                    collect($gradesAdd)
                        ->pluck('school_session_id')
                        ->unique()
                )
                ->get();

            $schoolSessionLessonIds = Arr::pluck(
                $schoolSessions
                    ->toArray(),
                'weekly_schedule.lesson_id',
                'id'
            );

            DB::beginTransaction();

            foreach ($gradesAdd as $gradeAdd) {
                $grade = Grade::create(
                    Arr::set(
                        $gradeAdd,
                        'lesson_id',
                        $schoolSessionLessonIds[$gradeAdd['school_session_id']]
                    )
                );
                $grade->refresh();
                $grades[] = $grade;
            }

            $user->appendLog($request, ['new' => $grades]);
            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'نمره'])
                ->setData('grades',  $grades)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'gradeStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'نمره'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteGradeRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $grade = Grade::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $grade->toArray()]);
            $grade->delete();


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'نمره'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'gradeDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'نمره'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateGradeRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $gradeOriginals = [];
            $user = User::find(auth()->user()->id ?? -1);
            $gradesUpdate = CommonTools::safeRequest($request, 'grades');

            $schoolSessions = SchoolSession::select(
                [
                    'id',
                    'weekly_schedule_id',
                ]
            )
                ->with(
                    [
                        'weeklySchedule:weekly_schedules.id,lesson_id'
                    ]
                )
                ->whereIn(
                    'id',
                    collect($gradesUpdate)
                        ->pluck('school_session_id')
                        ->unique()
                )
                ->get();

            $schoolSessionLessonIds = Arr::pluck(
                $schoolSessions
                    ->toArray(),
                'weekly_schedule.lesson_id',
                'id'
            );

            DB::beginTransaction();

            foreach ($gradesUpdate as $gradeUpdate) {
                $gradeOriginal = Grade::find($gradeUpdate['id']);

                $gradeOriginals[] = clone $gradeOriginal;

                $gradeOriginal->update(
                    Arr::set(
                        $gradeUpdate,
                        'lesson_id',
                        $schoolSessionLessonIds[$gradeUpdate['school_session_id']]
                    )
                );
            }

            $user->appendLog($request, [
                'old' => array_map(fn($value) => $value->toArray(), $gradeOriginals),
                'new' => $gradesUpdate,
            ]);

            DB::commit();

            foreach ($gradeOriginals as $gradeOriginal) {
                $gradeOriginal->refresh();
            }

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'نمره']
                )
                ->setData('grades',  $gradeOriginals)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'gradeUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'نمره'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
