<?php

namespace App\Http\Controllers;

use App\Enums\PermissionLessonAccess;
use App\Http\Requests\Lesson\DeleteLessonRequest;
use App\Http\Requests\Lesson\ExcelStoreLessonRequest;
use App\Http\Requests\Lesson\StoreLessonRequest;
use App\Http\Requests\Lesson\UpdateLessonRequest;
use App\Http\Requests\Lesson\ViewLessonRequest;
use App\Imports\LessonImport;
use App\Models\Lesson;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use DB;
use Excel;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class LessonController extends Controller
{

    public function index(ViewLessonRequest $request)
    {
        try {
            $user = User::find(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions([PermissionLessonAccess::view]);
            $dataTableTools = new DataTableTools();

            $query = Lesson::with(
                [
                    'classRoom:id,major_id,academic_level_id,title,code',
                    'classRoom.major',
                    'classRoom.academicLevel',
                ]
            )
                ->when(
                    $isNotAdmin,
                    fn($qLesson) =>
                    $qLesson->whereHas(
                        'classRoom',
                        fn($qClassRoom) =>
                        $qClassRoom->whereHas(
                            'weeklySchedules',
                            fn($qWeeklySchedule) =>
                            $qWeeklySchedule->where(
                                'teacher_id',
                                '=',
                                $user->id
                            )
                        )
                    )
                );

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('lessons', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'درس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreLessonRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $lessonAdd = CommonTools::safeRequest($request, 'lesson');


            DB::beginTransaction();

            $lesson = Lesson::create($lessonAdd);


            $user->appendLog($request, ['new' => $lesson]);
            DB::commit();

            $lesson->load(
                [
                    'classRoom:id,major_id,academic_level_id,title,code',
                    'classRoom.major',
                    'classRoom.academicLevel',
                ]
            );

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'درس'])
                ->setData('lesson', $lesson)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'lessonStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'درس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function excelStore(ExcelStoreLessonRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $lessonsAdd = CommonTools::safeRequest($request, 'xlsx');

            DB::beginTransaction();

            Excel::import(
                new LessonImport(),
                $lessonsAdd
            );

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_recorded',
                    ['title' => 'درس']
                )
                ->getJsonResponse();
        } catch (ValidationException $e) {
            DB::rollBack();
            CommonTools::registerException($e, 'lessonExcelStore');

            $messageErrors = $e->errors();

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage(array_slice($messageErrors, 0, min(15, count($messageErrors))))
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'lessonExcelStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'درس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteLessonRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $lesson = Lesson::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $lesson->toArray()]);
            $lesson->delete();


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'درس'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'lessonDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'درس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateLessonRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $user = User::find(auth()->user()->id ?? -1);
            $requestBody = CommonTools::safeRequest($request, 'id', 'lesson');
            $lessonUpdate = $requestBody['lesson'];
            $lessonOriginal = Lesson::find($requestBody['id']);

            DB::beginTransaction();

            $user->appendLog($request, [
                'old' => $lessonOriginal->toArray(),
                'new' => $lessonUpdate,
            ]);
            $lessonOriginal->update($lessonUpdate);

            DB::commit();

            $lessonOriginal->load(
                [
                    'classRoom:id,major_id,academic_level_id,title,code',
                    'classRoom.major',
                    'classRoom.academicLevel',
                ]
            );

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_updated', ['title' => 'درس'])
                ->setData('lesson', $lessonOriginal)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'lessonUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'درس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
