<?php

namespace App\Http\Controllers;

use App\Enums\PermissionMoralAccess;
use App\Http\Requests\Moral\DeleteMoralRequest;
use App\Http\Requests\Moral\StoreMoralRequest;
use App\Http\Requests\Moral\UpdateMoralRequest;
use App\Http\Requests\Moral\ViewMoralRequest;
use App\Http\Requests\Moral\ViewOwnChildrenMoralRequest;
use App\Models\Moral;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use DB;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class MoralController extends Controller
{

    public function index(ViewMoralRequest $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions(PermissionMoralAccess::view);

            $dataTableTools = new DataTableTools();

            $query = Moral::with([
                'student:id,first_name,last_name',
                'schoolSession:id,weekly_schedule_id,teacher_user_id,held,created_at,updated_at',
                'schoolSession.teacher:id',
                'schoolSession.teacher.profile:id,user_id,first_name,last_name'
            ])
                ->when($isNotAdmin, fn($qMoral) =>
                $qMoral->whereHas(
                    'schoolSession',
                    fn($qSchoolSession) =>
                    $qSchoolSession->where('teacher_user_id', '=', $user->id)
                ));

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('morals', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'وضعیت اخلاقی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }


    public function indexOwnChildren(ViewOwnChildrenMoralRequest $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);
            $dataTableTools = new DataTableTools();

            $query = Moral::with([
                'student:id,first_name,last_name',
                'schoolSession:id,weekly_schedule_id,teacher_user_id,held,created_at,updated_at',
                'schoolSession.teacher:id',
                'schoolSession.teacher.profile:id,user_id,first_name,last_name'
            ])
                ->whereHas(
                    'student',
                    fn($qStudent) =>
                    $qStudent->where('parent_id', '=', $user->id)
                );

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('morals', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'وضعیت اخلاقی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreMoralRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $moralsAdd = CommonTools::safeRequest($request, 'morals');
            $morals = [];

            DB::beginTransaction();

            Moral::upsertWithEncryption(
                $moralsAdd,
                ['school_session_id', 'student_id'],
            );

            DB::commit();

            $morals = Moral::where(
                fn($qMoral) =>
                collect($moralsAdd)
                    ->each(
                        fn($elMoral) =>
                        $qMoral->orWhere(
                            fn($qgMoral) =>
                            $qgMoral->where('school_session_id', '=', $elMoral['school_session_id'])
                                ->where('student_id', '=', $elMoral['student_id'])
                        )
                    )
            )
                ->get();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'وضعیت اخلاقی'])
                ->setData('morals',  $morals)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'moralStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'وضعیت اخلاقی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteMoralRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $moral = Moral::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $moral->toArray()]);
            $moral->delete();


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'وضعیت اخلاقی'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'moralDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'وضعیت اخلاقی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateMoralRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $moralOriginals = [];
            $user = User::find(auth()->user()->id ?? -1);
            $moralsUpdate = CommonTools::safeRequest($request, 'morals');

            DB::beginTransaction();

            foreach ($moralsUpdate as $moralUpdate) {
                if (!isset($moralUpdate['comment'])) {
                    $moralUpdate['comment'] = null;
                }

                $moralOriginal = Moral::find($moralUpdate['id']);

                $moralOriginals[] = clone $moralOriginal;

                $moralOriginal->update($moralUpdate);
            }

            $user->appendLog($request, [
                'old' => array_map(fn($value) => $value->toArray(), $moralOriginals),
                'new' => $moralsUpdate,
            ]);

            DB::commit();

            foreach ($moralOriginals as $moralOriginal) {
                $moralOriginal->refresh();
            }

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'وضعیت اخلاقی']
                )
                ->setData('morals',  $moralOriginals)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'moralUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'وضعیت اخلاقی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
