<?php

namespace App\Http\Controllers;

use App\Enums\ApplicationConfigActions;
use App\Enums\PermissionPresenceAccess;
use App\Http\Requests\Presence\DeletePresenceRequest;
use App\Http\Requests\Presence\ExportPDFPresenceRequest;
use App\Http\Requests\Presence\SendSmsPresenceRequest;
use App\Http\Requests\Presence\UpdatePresenceRequest;
use App\Http\Requests\Presence\ViewPresenceRequest;
use App\Models\ApplicationConfig;
use App\Models\Presence;
use App\Models\Profile;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use App\Tools\SmsTools;
use Arr;
use DB;
use PDF;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class PresenceController extends Controller
{

    public function index(ViewPresenceRequest $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions(PermissionPresenceAccess::view);


            $dataTableTools = new DataTableTools();
            $query = $isNotAdmin ? $user->presences() : Presence::select();

            $query->with([
                'student',
                'schoolSession',
            ]);

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('presences', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'حضور و غیاب']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }


    public function destroy(DeletePresenceRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $presence = Presence::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $presence->toArray()]);
            $presence->delete();


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_deleted',
                    ['title' => 'حضور و غیاب']
                )
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'presenceDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_deleting_information',
                ['title' => 'حضور و غیاب']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdatePresenceRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $user = User::find(auth()->user()->id ?? -1);
            $presenceUpdate = CommonTools::safeRequest($request,  'presence');
            $presenceStudents = Arr::pull($presenceUpdate, 'students');
            $schoolSessionId = $presenceUpdate['school_session_id'];

            DB::beginTransaction();

            $user->appendLog($request, [
                'new' => $presenceUpdate,
            ]);

            $updatedPresenceStudents = [];
            foreach ($presenceStudents as $ps) {
                $updatedPresenceStudents[] = Presence::updateOrCreate(
                    [
                        'school_session_id' => $schoolSessionId,
                        'student_id' => $ps['student_id']
                    ],
                    [
                        'student_id' => $ps['student_id'],
                        'status' => $ps['status'],
                        ...(
                            $ps['status'] === 1 ||
                            $ps['status'] === true ||
                            $ps['status'] === '1' ?
                            [
                                'entry_time' => $ps['entry_time'] ?? null,
                                'exit_time' => $ps['exit_time'] ?? null,
                                'delay_reason' => $ps['delay_reason'] ?? null,
                                'early_exit_reason' => $ps['early_exit_reason'] ?? null,

                            ] :
                            [
                                'entry_time' => null,
                                'exit_time' => null,
                                'delay_reason' => null,
                                'early_exit_reason' => null,
                            ]
                        )
                    ]
                );
            }


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'حضور و غیاب']
                )
                ->setData('presences',  $updatedPresenceStudents)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'presenceUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_updating_information',
                ['title' => 'حضور و غیاب']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function exportPDF(ExportPDFPresenceRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $dataTableTools = new DataTableTools();

            $applicationConfigs = ApplicationConfig::where(
                fn($qAppConfig) => $qAppConfig->whereEncrypted(
                    'title',
                    '=',
                    ApplicationConfigActions::schoolControllerInformationUpdate
                )
                    ->orWhereEncrypted(
                        'title',
                        '=',
                        ApplicationConfigActions::schoolControllerPrivateInformationUpdate
                    )
            )
                ->where(
                    'active',
                    '=',
                    true
                )
                ->limit(2)
                ->orderBy('title')
                ->get()
                ->toArray();

            [$schoolConfig, $schoolPrivateConfig] = $applicationConfigs;

            $query = Presence::with(['student']);

            $dataTableTools->doQuery($query, $request);
            $presenceQueryWrapFilters = (new DataTableTools())->wrapFilters($query, $request);
            $firstPresence = (clone $presenceQueryWrapFilters)->orderBy('created_at')->first();
            $lastPresence = (clone $presenceQueryWrapFilters)->orderByDesc('created_at')->first();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                $pdf = PDF::loadView('pdf.presences', [
                    'presences' => $dataTableTools->data,
                    'firstPresence' => $firstPresence,
                    'lastPresence' => $lastPresence,
                    'reportInfo' => [
                        'school_code' => $schoolPrivateConfig['value']['school_code'],
                        'school_type' => $schoolPrivateConfig['value']['school_type'],
                        'school_title' => $schoolConfig['value']['title'],
                        'school_province' => $schoolConfig['value']['province'],
                        'school_district' => $schoolConfig['value']['district'],
                    ]
                ]);

                return $pdf->stream('document.pdf');
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'حضور و غیاب']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function sendSms(SendSmsPresenceRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {

            $formBody = CommonTools::safeRequest($request, 'presence_ids', 'message');
            $presenceIds = $formBody['presence_ids'];
            $message = $formBody['message'];
            $messageVars = SmsTools::getVariables($message);
            $hasMessageVars = $messageVars->count() > 0;

            $phones = Profile::select(
                [
                    'id',
                    'user_id',
                    'phone',
                    ...(
                        $messageVars->contains(':parent_last_name') ||
                        $messageVars->contains(':parent_full_name')
                        ? [
                            'first_name',
                            'last_name'
                        ] : []
                    )
                ]
            )
                ->whereHas(
                    'children',
                    fn($qChildren) =>
                    $qChildren->whereHas(
                        'presences',
                        fn($qPresences) =>
                        $qPresences->whereIn('id', $presenceIds)
                    )
                )
                ->get();

            if (!$hasMessageVars) {
                $phones = $phones->pluck('phone');
            } else {
                $phones = $phones->flatMap(fn($v) => [
                    $v->phone => [
                        ...($messageVars->contains(':parent_last_name') ? [
                            ':parent_last_name' => $v->last_name
                        ] : []),
                        ...($messageVars->contains(':parent_full_name') ? [
                            ':parent_full_name' => $v->first_name . ' ' . $v->last_name
                        ] : [])
                    ]
                ]);
            }


            SmsTools::sendFormatSms($phones, $message);

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_SUCCESS)
                ->setMessage(
                    __('messages.information_was_successfully_send_sms', ['title' => 'حضور و غیاب'])
                )
                ->getJsonResponse();
        } catch (Throwable $e) {
            CommonTools::registerException($e, 'presenceSendSms');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessage(
                __(
                    'messages.error_send_sms',
                    ['title' => 'حضور و غیاب']
                )
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
