<?php

namespace App\Http\Controllers;

use App\Enums\PermissionSchoolSessionAccess;
use App\Exceptions\AnyException;
use App\Http\Requests\SchoolSession\DeleteSchoolSessionRequest;
use App\Http\Requests\SchoolSession\StoreSchoolSessionRequest;
use App\Http\Requests\SchoolSession\UpdateSchoolSessionRequest;
use App\Http\Requests\SchoolSession\ViewSchoolSessionRequest;
use App\Models\SchoolSession;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use Arr;
use Carbon\Carbon;
use DB;
use Morilog\Jalali\Jalalian;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class SchoolSessionController extends Controller
{

    public function index(ViewSchoolSessionRequest $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);

            $isNotAdmin = !$user->hasPermissions([
                PermissionSchoolSessionAccess::view->value
            ]);

            $dataTableTools = new DataTableTools();
            $query = SchoolSession::with([
                'weeklySchedule',
                'weeklySchedule.bell',
                'weeklySchedule.lesson',
                'weeklySchedule.classRoom',
                'weeklySchedule.classRoom.major',
                'weeklySchedule.classRoom.academicLevel',
                'teacher:id',
                'teacher.profile:id,user_id,first_name,last_name',
                'teacher.teacher:id,user_id,personally_code',
            ])
                ->when($isNotAdmin, function ($qSchoolSession) use ($user) {
                    $qSchoolSession->where('teacher_user_id', '=', $user->id);
                });

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('school_sessions', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'برگزاری کلاس']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function currentIndex(ViewSchoolSessionRequest $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);

            [$schoolSession] = self::getSchoolSessionWithDataTime($user);

            $schoolSession->load([
                'weeklySchedule',
                'weeklySchedule.bell',
                'weeklySchedule.lesson',
                'weeklySchedule.classRoom',
                'weeklySchedule.classRoom.major',
                'weeklySchedule.classRoom.academicLevel',
            ]);

            if ($schoolSession == null) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessage(__('messages.information_was_successfully_no_content'))
                    ->getJsonResponse(Response::HTTP_BAD_REQUEST);
            }

            return ResponseTools::getInstance()
                ->setMessage(__('messages.information_was_successfully_get'))
                ->setData('school_session', $schoolSession)
                ->getJsonResponse();
        } catch (AnyException $e) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($e->getMessage())
                ->getJsonResponse($e->getCode());
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'برگزاری کلاس']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreSchoolSessionRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::findOrFail(auth()->user()->id ?? -1);
            $schoolSessionAdd = CommonTools::safeRequest($request, 'school_session') ?? [];
            $adminAccess = $user->hasPermissions([PermissionSchoolSessionAccess::update]);
            $teacherUserId = $adminAccess === true ?
                Arr::pull($schoolSessionAdd, 'teacher_user_id') :
                null;

            if ($teacherUserId != null) {
                $user = User::findOrFail($teacherUserId);
            }

            DB::beginTransaction();

            [$schoolSession, $isCreated] = self::createOrFirstSchoolSession(
                $user,
                $schoolSessionAdd,
                $adminAccess
            );

            if (!$isCreated) {
                $schoolSession->update($schoolSessionAdd);
            }

            $user->appendLog($request, ['new' => $schoolSession]);

            DB::commit();

            $schoolSession->load([
                'weeklySchedule'
            ]);

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_recorded',
                    ['title' => 'برگزاری کلاس']
                )
                ->setData('school_session', $schoolSession)
                ->getJsonResponse();
        } catch (AnyException $e) {
            DB::rollBack();

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($e->getMessage())
                ->getJsonResponse($e->getCode());
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'schoolSessionStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_recording_information',
                ['title' => 'برگزاری کلاس']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteSchoolSessionRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $schoolSession = SchoolSession::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $schoolSession->toArray()]);
            $schoolSession->delete();

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'برگزاری کلاس'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'schoolSessionDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'برگزاری کلاس'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateSchoolSessionRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $user = User::find(auth()->user()->id ?? -1);
            $requestBody = CommonTools::safeRequest($request, 'id', 'school_session');
            $schoolSessionUpdate = $requestBody['school_session'];
            $adminAccess = $user->hasPermissions([PermissionSchoolSessionAccess::update]);
            $teacherUserId = $adminAccess === true  ?
                Arr::pull($schoolSessionUpdate, 'teacher_user_id') :
                null;

            if ($teacherUserId != null) {
                $user = User::findOrFail($teacherUserId);
            }

            DB::beginTransaction();


            [$schoolSessionOriginal] = self::createOrFirstSchoolSession(
                $user,
                array_merge(
                    $schoolSessionUpdate,
                    ['id' => $requestBody['id']]
                ),
                $adminAccess
            );

            if ($schoolSessionOriginal == null) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessageFormat(
                        'messages.access_denied_to',
                        ['title' => 'برگزاری کلاس']
                    )
                    ->getJsonResponse(Response::HTTP_BAD_REQUEST);
            }

            $user->appendLog($request, [
                'old' => $schoolSessionOriginal->toArray(),
                'new' => $schoolSessionUpdate,
            ]);

            $schoolSessionOriginal->update($schoolSessionUpdate);

            DB::commit();


            $schoolSessionOriginal->load([
                'weeklySchedule'
            ]);

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'برگزاری کلاس']
                )
                ->setData('school_session', $schoolSessionOriginal)
                ->getJsonResponse();
        } catch (AnyException $e) {
            DB::rollBack();

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($e->getMessage())
                ->getJsonResponse($e->getCode());
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'schoolSessionUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_updating_information',
                ['title' => 'برگزاری کلاس']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * get school session with startedAt
     * @param User $user
     * @param mixed $startedAt
     * @throws AnyException
     * @return array{SchoolSession|null,\App\Models\WeeklySchedule,Carbon}|null
     */
    public static function getSchoolSessionWithDataTime(User $user, $startedAt = null): array
    {
        $startedAtCarbon = $startedAt != null ?
            Carbon::createFromFormat(
                'Y-m-d H:i',
                $startedAt
            ) :
            new Carbon();

        $startedAtCarbonJalali = Jalalian::fromCarbon(
            $startedAtCarbon->tz('Asia/Tehran')
        );

        $dayOfWeek = $startedAtCarbonJalali->getDayOfWeek();
        $hour = $startedAtCarbonJalali->format('H:i');

        $weeklySchedule = $user->weeklySchedules()
            ->where('day', '=', $dayOfWeek)
            ->whereHas('bell', function ($qBell) use ($hour) {
                $qBell->whereTime('started_at', '<=', $hour)
                    ->whereTime('ended_at', '>=', $hour);
            })
            ->orderBy('id', 'desc')
            ->first();

        if ($weeklySchedule == null) {
            throw new AnyException(
                __('messages.error_not_have_weekly_schedule_bell'),
                Response::HTTP_BAD_REQUEST
            );
        }

        $bell = $weeklySchedule->bell()->firstOrFail();

        $schoolSession = SchoolSession::where(
            'teacher_user_id',
            '=',
            $user->id
        )
            ->whereDate(
                'created_at',
                '=',
                $startedAtCarbon->format('Y-m-d')
            )
            ->whereTime('created_at', '>=', $bell->started_at)
            ->whereTime('created_at', '<=', $bell->ended_at)
            ->orderBy('id', 'desc')
            ->first();

        return [$schoolSession, $weeklySchedule, $startedAtCarbon];
    }

    /**
     * get or create schoolSession with DataTime or current DataTime
     * @template TModel
     * @param User $user
     * @param array $schoolSessionAdd
     * @return array{SchoolSession|TModel|\Illuminate\Database\Eloquent\Collection<int, TModel>,bool}
     */
    public static function createOrFirstSchoolSession(User $user, array $schoolSessionAdd, $adminAccess = false)
    {
        $schoolSessionId = Arr::pull($schoolSessionAdd, 'id');

        if ($schoolSessionId != null) {
            if ($adminAccess) {
                return SchoolSession::find($schoolSessionId);
            }

            return [
                $user->schoolSessions()
                    ->find($schoolSessionId),
                false
            ];
        }

        $startedAt = Arr::pull($schoolSessionAdd, 'started_at');


        [
            $schoolSession,
            $weeklySchedule,
            $startedAtCarbon
        ] = self::getSchoolSessionWithDataTime($user, $startedAt);

        if ($schoolSession != null) {
            return [$schoolSession, false];
        }

        return [
            $user->schoolSessions()
                ->create(
                    array_merge(
                        ['held' => true],
                        $schoolSessionAdd,
                        [
                            'weekly_schedule_id' => $weeklySchedule->id,
                            'created_at' => $startedAtCarbon
                        ]
                    )
                ),
            true
        ];
    }
}
