<?php

namespace App\Http\Controllers;

use App\Http\Requests\Teacher\DeleteTeacherRequest;
use App\Http\Requests\Teacher\ExcelStoreTeacherRequest;
use App\Http\Requests\Teacher\ShowTeacherRequest;
use App\Http\Requests\Teacher\StoreTeacherRequest;
use App\Http\Requests\Teacher\UpdateTeacherRequest;
use App\Http\Requests\Teacher\ViewTeacherRequest;
use App\Imports\TeacherImport;
use App\Models\Teacher;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use Excel;
use Exception;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\DB;
use Throwable;
use Maatwebsite\Excel\Validators\ValidationException;

class TeacherController extends Controller
{

    public function index(ViewTeacherRequest $request)
    {
        try {
            $dataTableTools = new DataTableTools();
            $query = Teacher::with([
                'user:id,name,created_at',
                'user.profile:user_id,first_name,last_name,address,national_code,phone',
                'user.primaryRole',
            ]);

            $dataTableTools->doQuery($query, $request)
                ->appendFields(
                    [
                        'user.active',
                    ]
                )
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('teachers', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'معلم'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function show(ShowTeacherRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $id = CommonTools::safeRequest($request, 'id');

            $fetchTeacher = Teacher::with([
                'user:id,name,created_at',
                'user.profile',
                'user.primaryRole',
            ])
                ->where('id', $id)
                ->first();

            if ($fetchTeacher) {
                $fetchTeacher
                    ->user
                    ->append([
                        'active',
                    ]);

                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setData('teacher', $fetchTeacher)
                    ->getJsonResponse();
            }
        } catch (Exception $exception) {
            CommonTools::registerException($exception, 'teacherShow');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'معلم'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreTeacherRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        $file = null;

        try {
            $teacherAdd = CommonTools::safeRequest($request, 'teacher');

            DB::beginTransaction();

            if (!isset($teacherAdd['user']['password'])) {
                $teacherAdd['user']['password'] = $teacherAdd['personally_code'];
            }

            [$user, $password] = UserController::createUser(
                $teacherAdd['user'],
                $file,
                'teacher'
            );

            $teacher = $user->teacher()->create([
                'personally_code' => $teacherAdd['personally_code']
            ]);

            DB::commit();

            $teacher->load([
                'user:id,name,created_at',
                'user.profile',
                'user.primaryRole',
            ]);

            $teacher
                ->user
                ->append([
                    'active',
                ]);

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_recorded',
                    ['title' => 'معلم']
                )
                ->setData('teacher', array_merge($teacher->toArray(), ['password' => $password]))
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            if (isset($file)) {
                $file->removeFile();
            }

            CommonTools::registerException($e, 'teacherPersonnelStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_recording_information',
                ['title' => 'معلم']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function excelStore(ExcelStoreTeacherRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $teachersAdd = CommonTools::safeRequest($request, 'xlsx');

            DB::beginTransaction();

            Excel::import(
                new TeacherImport(),
                $teachersAdd
            );

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'معلم'])
                ->getJsonResponse();
        } catch (ValidationException $e) {
            DB::rollBack();
            CommonTools::registerException($e, 'teacherExcelStore');

            $messageErrors = $e->errors();

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage(array_slice($messageErrors, 0, min(15, count($messageErrors))))
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'teacherExcelStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'معلم'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteTeacherRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $userAdmin = User::find(auth()->user()->id ?? -1);
            //Check that the working account is active.
            $userAdmin->roles()
                ->where('primary', 1)
                ->firstOrFail();

            $teacher = Teacher::find($request->get('id', -1));
            $userOriginal = $teacher->user()->firstOrFail();

            DB::beginTransaction();

            UserController::deleteUser($userOriginal);

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_deleted',
                    ['title' => 'دسترسی']
                )
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'teacherDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_deleting_information',
                ['title' => 'دسترسی']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateTeacherRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        $fileUpdate = null;

        try {

            if (
                ! ($user = User::find(auth()->user()->id ?? -1)) ||
                ! ($access = $user->getAccess()) ||
                $access == User::ACCESS_DENIED ||
                $access == User::ACCESS_ADMIN_LIMITED ||
                $access == User::ACCESS_ADMINISTRATOR_LIMITED
            ) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessageFormat('messages.access_denied_to_page')
                    ->getJsonResponse(Response::HTTP_UNAUTHORIZED);
            }

            $requestBody = CommonTools::safeRequest($request, 'id', 'teacher');
            $teacherUpdate = $requestBody['teacher'];

            if ($access != User::ACCESS_PERSONAL_ACCOUNT && $access != User::ACCESS_PERSONAL_ACCOUNT_LIMITED) {

                //Check that the working account is active.
                $user->roles()
                    ->where('primary', 1)
                    ->firstOrFail();

                $teacherOriginal = Teacher::findOrFail($requestBody['id']);
                $userOriginal = $teacherOriginal->user()->firstOrFail();
                unset($requestBody['id']);
            } else {
                $userOriginal = $user;
                $teacherOriginal = $user->teacher()->firstOrFail();
            }

            if (!isset($teacherUpdate['user']['role_id'])) {
                $teacherUpdate['user']['role_id'] = $userOriginal
                    ->primaryRole()
                    ->first()
                    ->id;
            }

            DB::beginTransaction();

            UserController::updateUser(
                $user,
                $userOriginal,
                $teacherUpdate['user'],
                $fileUpdate
            );

            $teacherOriginal->update([
                'personally_code' => $teacherUpdate['personally_code']
            ]);

            DB::commit();

            $teacherOriginal->load([
                'user:id,name,created_at',
                'user.profile',
                'user.primaryRole',
            ]);

            $teacherOriginal
                ->user
                ->append([
                    'active',
                ]);

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'معلم']
                )
                ->setData('teacher', $teacherOriginal)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            if ($fileUpdate) {
                $fileUpdate->removeFile();
            }

            CommonTools::registerException($e, 'teacherUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'معلم'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
