<?php

namespace App\Http\Controllers;

use App\Enums\ApplicationConfigActions;
use App\Enums\PermissionTeachingReportAccess;
use App\Exceptions\AnyException;
use App\Http\Requests\TeachingReport\ExportPDFTeachingReportRequest;
use App\Http\Requests\TeachingReport\DeleteTeachingReportRequest;
use App\Http\Requests\TeachingReport\StoreTeachingReportRequest;
use App\Http\Requests\TeachingReport\UpdateTeachingReportRequest;
use App\Http\Requests\TeachingReport\ViewTeachingReportRequest;
use App\Models\ApplicationConfig;
use App\Models\ClassRoom;
use App\Models\Lesson;
use App\Models\Student;
use App\Models\TeachingReport;
use App\Models\User;
use App\Tools\ApplicationConfigTools;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use App\Tools\TermTools;
use DB;
use Morilog\Jalali\Jalalian;
use PDF;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class TeachingReportController extends Controller
{

    public function index(ViewTeachingReportRequest $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions(PermissionTeachingReportAccess::view);
            $dataTableTools = new DataTableTools();

            $query = TeachingReport::with([
                'student:id,first_name,last_name',
                'lesson:id,title',
                'classRoom:class_rooms.id,class_rooms.title',
            ])
                ->when(
                    $isNotAdmin,
                    fn($qTeachingReport) =>
                    $qTeachingReport->whereHas(
                        'lesson',
                        fn($qLesson) =>
                        $qLesson->whereHas(
                            'weeklySchedules',
                            fn($qWeeklySchedules) =>
                            $qWeeklySchedules
                                ->where('teacher_id', '=', $user->id)
                        )
                    )
                );

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('teaching_reports', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'نمره'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreTeachingReportRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $teachingReportsAdd = CommonTools::safeRequest($request, 'teaching_reports');
            $teachingReports = [];

            $teachingReportsAdd = $this->cleanPodmanyGrades($teachingReportsAdd);

            DB::beginTransaction();

            foreach ($teachingReportsAdd as $teachingReportAdd) {
                $teachingReport = TeachingReport::create($teachingReportAdd);
                $teachingReport->refresh();
                $teachingReports[] = $teachingReport;
            }

            $user->appendLog($request, ['new' => $teachingReports]);
            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'نمره'])
                ->setData('teaching_reports',  $teachingReports)
                ->getJsonResponse();
        } catch (AnyException $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'teachingReportStore');

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($e->getMessage())
                ->getJsonResponse($e->getCode());
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'teachingReportStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'نمره'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteTeachingReportRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $teachingReport = TeachingReport::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $teachingReport->toArray()]);
            $teachingReport->delete();


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'نمره'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'teachingReportDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'نمره'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateTeachingReportRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $teachingReportOriginals = [];
            $user = User::find(auth()->user()->id ?? -1);
            $teachingReportsUpdate = CommonTools::safeRequest($request, 'teaching_reports');

            $teachingReportsUpdate = $this->cleanPodmanyGrades($teachingReportsUpdate);

            DB::beginTransaction();

            foreach ($teachingReportsUpdate as $teachingReportUpdate) {
                $teachingReportOriginal = TeachingReport::find($teachingReportUpdate['id']);

                $teachingReportOriginals[] = clone $teachingReportOriginal;

                $teachingReportOriginal->update($teachingReportUpdate);
            }

            $user->appendLog($request, [
                'old' => array_map(fn($value) => $value->toArray(), $teachingReportOriginals),
                'new' => $teachingReportsUpdate,
            ]);

            DB::commit();

            foreach ($teachingReportOriginals as $teachingReportOriginal) {
                $teachingReportOriginal->refresh();
            }

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'نمره']
                )
                ->setData('teaching_reports',  $teachingReportOriginals)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'teachingReportUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'نمره'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function exportPDF(ExportPDFTeachingReportRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $dataTableTools = new DataTableTools();
            $requestBody = CommonTools::safeRequest($request, 'year', 'term');
            $year = $requestBody['year'] ?? Jalalian::now()->getYear();
            $term = $requestBody['term'] ?? $this->getCurrentTerm();

            $applicationConfigs = ApplicationConfig::where(
                fn($qAppConfig) => $qAppConfig->whereEncrypted(
                    'title',
                    '=',
                    ApplicationConfigActions::schoolControllerInformationUpdate
                )
                    ->orWhereEncrypted(
                        'title',
                        '=',
                        ApplicationConfigActions::schoolControllerPrivateInformationUpdate
                    )
                    ->orWhereEncrypted(
                        'title',
                        '=',
                        ApplicationConfigActions::schoolControllerPrivateTermUpdate
                    )
            )
                ->where(
                    'active',
                    '=',
                    true
                )
                ->get();

            $applicationConfigs = ApplicationConfigTools::mapKey($applicationConfigs);

            $schoolConfig = $applicationConfigs
                ->get(ApplicationConfigActions::schoolControllerInformationUpdate->value)
                ->toArray();
            $schoolPrivateConfig = $applicationConfigs
                ->get(ApplicationConfigActions::schoolControllerPrivateInformationUpdate->value)
                ->toArray();
            $schoolPrivateTerm = $applicationConfigs
                ->get(ApplicationConfigActions::schoolControllerPrivateTermUpdate->value)
                ->toArray();

            $termData = collect($schoolPrivateTerm['value'])->first(fn($el) => $el['term'] == $term);

            if (!$termData) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessage(__('messages.not_found', ['title' => 'نوبت']))
                    ->getJsonResponse(Response::HTTP_BAD_REQUEST);
            }

            $termDates = TermTools::getDateFromTerm($termData, $year);

            $query = TeachingReport::with([
                'lesson:id,class_room_id,title,is_podmany,passing_grade',
            ])
                ->where(
                    'term',
                    '=',
                    $term,
                )
                ->whereDate(
                    'created_at',
                    '>=',
                    $termDates['started_at'],
                )
                ->whereDate(
                    'created_at',
                    '<=',
                    $termDates['ended_at'],
                );

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            $teachingReports = collect($dataTableTools->data)
                ->groupBy('student_id');

            $studentIds = $teachingReports->keys();

            $students = Student::select([
                'id',
                'parent_id',
                'first_name',
                'last_name',
                'national_code',
            ])
                ->with([
                    'studentParent:id',
                    'studentParent.profile:id,user_id,first_name,last_name',
                ])
                ->whereIn('id', $studentIds)
                ->get();

            $studentGrades = $teachingReports->map(
                fn($elTR, $studentId) => array_merge(
                    $students->findOrFail($studentId)->toArray(),
                    [
                        'teaching_reports' => $elTR->toArray()
                    ]
                )
            )
                ->values();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {

                $studentGrades = $this->mapStudentGrades(
                    $studentGrades,
                    $termDates
                );

                $pdf = PDF::loadView('pdf.teaching-reports', [
                    'studentGrades' => $studentGrades->toArray(),
                    'term' => $term,
                    'year' => $year,
                    'reportInfo' => [
                        'school_code' => $schoolPrivateConfig['value']['school_code'],
                        'school_type' => $schoolPrivateConfig['value']['school_type'],
                        'school_title' => $schoolConfig['value']['title'],
                        'school_province' => $schoolConfig['value']['province'],
                        'school_district' => $schoolConfig['value']['district'],
                    ]
                ]);

                return $pdf->stream('document.pdf');
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            CommonTools::registerException($e, 'gradeExportPDF');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'چاپ نمره نهایی']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    private function getCurrentTerm()
    {
        $ApplicationConfig = ApplicationConfig::whereEncrypted(
            'title',
            '=',
            CommonTools::valueOfEnum(
                ApplicationConfigActions::schoolControllerPrivateTermUpdate
            )
        )
            ->firstOrFail();


        $term = TermTools::getCurrentTerm($ApplicationConfig->value);

        if ($term == null) {
            throw new AnyException(
                __('messages.error_not_found_term'),
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        return intval($term['term']);
    }

    /**
     * set null for other grades
     * @param array<array{lesson_id:int,final_grade:float,term:int,genesis_grade:float,podmany_one_grade:float,podmany_two_grade:float,podmany_three_grade:float,podmany_four_grade:float,podmany_five_grade:float}> $teachingReports
     * @return array
     */
    private function cleanPodmanyGrades($teachingReports)
    {
        $collections = collect($teachingReports);
        $lessonIds = $collections->pluck('lesson_id')
            ->unique();

        $lessons = Lesson::select(['id', 'is_podmany'])
            ->whereIn('id', $lessonIds)
            ->get();

        $termOnePodmany = array_fill_keys(TeachingReport::TERM_ONE_PODMANY, null);
        $termTwoPodmany = array_fill_keys(TeachingReport::TERM_TWO_PODMANY, null);
        $notPodmany = array_fill_keys(TeachingReport::NOT_PODMANY, null);


        foreach ($teachingReports as $idx => $teachingReport) {
            $lesson = $lessons->findOrFail($teachingReport['lesson_id']);
            $term = $teachingReport['term'] ?? 1;
            $isPodmany = ($lesson?->is_podmany ?? false) == true;

            if ($isPodmany) {
                $teachingReports[$idx] = array_merge(
                    $teachingReport,
                    $term == 1 ?
                        $termOnePodmany : (
                            $term == 2 ?
                            $termTwoPodmany :
                            $notPodmany
                        )
                );
            } else {
                $teachingReports[$idx] = array_merge(
                    $teachingReport,
                    $notPodmany
                );
            }
        }

        return $teachingReports;
    }

    /**
     * append class_room and teachers to studentGrades
     * @param \Illuminate\Support\Collection $studentGrades
     * @param array{started_at: \Carbon\Carbon, ended_at: \Carbon\Carbon} $termDates
     * @return \Illuminate\Support\Collection
     */
    private function mapStudentGrades($studentGrades, $termDates)
    {
        $classRoomIds = $studentGrades
            ->pluck('teaching_reports.*.lesson.class_room_id')
            ->flatten(1)
            ->values()
            ->unique();

        $classRooms = ClassRoom::with(
            [
                'academicLevel',
                'major'
            ]
        )
            ->whereIn('id', $classRoomIds)
            ->get();

        $studentGrades = $studentGrades->map(
            fn($elSG) =>
            array_merge(
                $elSG,
                [
                    'class_room' =>
                    optional(
                        $classRooms->find(
                            data_get($elSG, 'teaching_reports.0.lesson.class_room_id')
                        )
                    )
                        ->toArray(),

                    'teaching_reports' => collect($elSG['teaching_reports'])
                        ->map(
                            fn($elTR) =>
                            data_set(
                                $elTR,
                                'lesson.teachers',
                                User::select(['id'])
                                    ->with(['profile:id,user_id,first_name,last_name'])
                                    ->whereHas(
                                        'weeklySchedules',
                                        fn($qWeeklySchedules) =>
                                        $qWeeklySchedules->where(
                                            'lesson_id',
                                            '=',
                                            $elTR['lesson_id']
                                        )
                                            ->whereDate(
                                                'created_at',
                                                '>=',
                                                $termDates['started_at'],
                                            )
                                            ->whereDate(
                                                'created_at',
                                                '<=',
                                                $termDates['ended_at'],
                                            )
                                    )
                                    ->get()
                                    ->toArray()
                            )
                        )
                        ->sortBy([
                            fn($a, $b) => $a['lesson']['is_podmany'] <=> $b['lesson']['is_podmany'],
                            fn($a, $b) => $a['lesson']['id'] <=> $b['lesson']['id'],
                        ])
                        ->values()
                        ->toArray()
                ]
            )
        );

        return $studentGrades;
    }
}
