<?php

namespace App\Http\Controllers;

use App\Http\Requests\User\ChangeUserPasswordRequest;
use App\Http\Requests\User\DeleteUserRequest;
use App\Http\Requests\User\ShowSelfUserRequest;
use App\Http\Requests\User\ShowUserRequest;
use App\Http\Requests\User\UpdateUserStatusRequest;
use App\Http\Requests\User\StoreUserRequest;
use App\Http\Requests\User\UpdateUserRequest;
use App\Http\Requests\User\ViewUserRequest;
use App\Models\Image;
use App\Models\Role;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\FileTools;
use App\Tools\RandomGenerator;
use App\Tools\ResponseTools;
use Exception;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\DB;
use Throwable;

class UserController extends Controller
{

    public function index(ViewUserRequest $request)
    {
        try {
            $dataTableTools = new DataTableTools();
            $query = User::select(['name', 'id', 'created_at'])
                ->with([
                    'profile:user_id,first_name,last_name,national_code,address,phone',
                    'primaryRole',
                ]);

            $dataTableTools->doQuery($query, $request)
                ->appendFields(
                    [
                        'active',
                    ]
                )
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('users', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'کاربر'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function show(ShowUserRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $id = CommonTools::safeRequest($request, 'id');

            $fetchUser = User::with([
                'profile',
                'profile.image',
                'primaryRole',
            ]);

            $fetchUser = $fetchUser->where('id', $id)
                ->first();

            if ($fetchUser) {
                $fetchUser
                    ->append([
                        'active',
                    ]);

                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setData('user', $fetchUser)
                    ->getJsonResponse();
            }
        } catch (Exception $exception) {
            CommonTools::registerException($exception, 'userShow');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'کاربر'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function me(ShowSelfUserRequest $request)
    {
        try {
            $user = User::with([
                'profile',
                'profile.image',
                'primaryRole.permissions'
            ])
                ->find(auth()->id());

            if ($user) {
                $user->append(
                    [
                        'active',
                    ]
                );

                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setData('user', $user)
                    ->getJsonResponse();
            }
        } catch (Exception $exception) {
            CommonTools::registerException($exception, 'userShow');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'کاربر'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreUserRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        $file = null;

        try {
            $userAdd = CommonTools::safeRequest($request, 'user');

            DB::beginTransaction();

            [$user, $password] = self::createUser($userAdd, $file);

            DB::commit();

            $user->load([
                'profile',
                'profile.image',
                'primaryRole.permissions'
            ]);

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_recorded',
                    ['title' => 'کاربر']
                )
                ->setData('user', array_merge(
                    $user->toArray(),
                    ['password' => $password]
                ))
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            if (isset($file)) {
                $file->removeFile();
            }

            CommonTools::registerException($e, 'userPersonnelStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_recording_information',
                ['title' => 'کاربر']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function statusUpdate(UpdateUserStatusRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }
        try {
            $userAdmin = User::find(auth()->user()->id ?? -1);
            //Check that the working account is active.
            $userAdmin->roles()
                ->where('primary', 1)
                ->firstOrFail();

            $requestBody = CommonTools::safeRequest($request, 'id', 'user');
            $userUpdate = $requestBody['user'];
            $user = User::find($requestBody['id']);

            $userRole = $user->userRoles()->first();

            if ($userRole) {
                DB::beginTransaction();

                $userRole->primary = $userUpdate['active'] ?? 0;
                $userRole->save();
                $userRole->touch();

                DB::commit();

                return ResponseTools::getInstance()
                    ->setMessageFormat(
                        'messages.information_was_successfully_updated',
                        ['title' => 'کاربر']
                    )
                    ->getJsonResponse();
            }
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'userStatusUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_updating_information',
                ['title' => 'کاربر']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteUserRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $userAdmin = User::find(auth()->user()->id ?? -1);
            //Check that the working account is active.
            $userAdmin->roles()
                ->where('primary', 1)
                ->firstOrFail();

            $user = User::find($request->get('id', -1));

            DB::beginTransaction();

            self::deleteUser($user);

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_deleted',
                    ['title' => 'دسترسی']
                )
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'userDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_deleting_information',
                ['title' => 'دسترسی']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateUserRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        $fileUpdate = null;

        try {
            if (
                ! ($user = User::find(auth()->user()->id ?? -1)) ||
                ! ($access = $user->getAccess()) ||
                $access == User::ACCESS_DENIED ||
                $access == User::ACCESS_ADMIN_LIMITED ||
                $access == User::ACCESS_ADMINISTRATOR_LIMITED
            ) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessageFormat('messages.access_denied_to_page')
                    ->getJsonResponse(Response::HTTP_UNAUTHORIZED);
            }

            $requestBody = CommonTools::safeRequest($request, 'id', 'user');
            $userUpdate = $requestBody['user'];

            $userOriginal = $user;

            if ($access != User::ACCESS_PERSONAL_ACCOUNT && $access != User::ACCESS_PERSONAL_ACCOUNT_LIMITED) {

                //Check that the working account is active.
                $user->roles()
                    ->where('primary', 1)
                    ->firstOrFail();

                $userOriginal = User::findOrFail($requestBody['id']);
                unset($requestBody['id']);
            }

            DB::beginTransaction();

            self::updateUser($user, $userOriginal, $userUpdate, $fileUpdate);

            DB::commit();

            $userOriginal->load([
                'profile',
                'profile.image',
                'primaryRole.permissions'
            ]);
            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'کاربر']
                )
                ->setData('user', $userOriginal)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            if ($fileUpdate) {
                $fileUpdate->removeFile();
            }

            CommonTools::registerException($e, 'userUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_updating_information',
                ['title' => 'کاربر']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function changePassword(ChangeUserPasswordRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            if (
                ! ($user = User::find(auth()->user()->id ?? -1)) ||
                ! ($access = $user->getAccess()) ||
                $access == User::ACCESS_DENIED ||
                $access == User::ACCESS_ADMIN_LIMITED ||
                $access == User::ACCESS_ADMINISTRATOR_LIMITED ||
                $access == User::ACCESS_ADMINISTRATOR
            ) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessageFormat('messages.access_denied_to_page')
                    ->getJsonResponse(Response::HTTP_UNAUTHORIZED);
            }

            $userUpdate = CommonTools::safeRequest($request, 'user');

            if (!password_verify($userUpdate['current_password'], $user->password)) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessageFormat('messages.password_is_incorrect')
                    ->getJsonResponse(Response::HTTP_BAD_REQUEST);
            }

            DB::beginTransaction();

            $user->password = $userUpdate['password'];

            $user->save();
            $user->touch();

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'تعویض رمز کاربر']
                )
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'userChangePassword');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_updating_information',
                ['title' => 'تعویض رمز کاربر']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }


    /**
     * createUser
     * @param mixed $userAdd user of request
     * @param mixed $file file for image profile
     * @param 
     * @return array{User, string}
     */
    public static function createUser($userAdd, &$file, $roleName = 'personalAccount')
    {
        $userAdmin = User::find(auth()->user()->id ?? -1);
        //Check that the working account is active.
        $userAdmin->roles()
            ->where('primary', 1)
            ->firstOrFail();

        $password = $userAdd['password'] ?? RandomGenerator::password();
        $personRoleId = Role::select("id")
            ->where('name', $roleName)
            ->firstOrFail()
            ->id;

        if (isset($userAdd['profile']['image'])) {
            $file = (new FileTools())->uploadFile(
                FileTools::$STORE_USER_PROFILE,
                $userAdd['profile']['image']
            )
                ->store();
        }

        $user = User::create(
            [
                'name' => $userAdd['name'],
                'password' => $password,
            ]
        );

        $profile = $user->profile()
            ->create($userAdd['profile']);

        if (isset($file)) {
            $profile->image()
                ->create([
                    'name' => $file->getFullName(),
                    'extra' => Image::$EXTRA_IMAGE_USER_PROFILE,
                ]);
        }

        $isActive = $userAdmin?->hasPermissions(['userStatusEdit'])
            && isset($userAdd['active']) ? $userAdd['active'] : 0;

        $user->roles()
            ->attach(
                (int) ($userAdd['role_id'] ?? $personRoleId),
                [
                    'primary' => $isActive,
                ]
            );

        return [$user, $password];
    }

    /**
     * updateUser
     * @param mixed $user user admin
     * @param mixed $userOriginal user update
     * @param mixed $userUpdate request user update
     * @param mixed $fileUpdate file update
     * @return void
     */
    public static function updateUser($user, $userOriginal, $userUpdate, &$fileUpdate)
    {
        $access = $user->getAccess();

        if (isset($userUpdate['profile']['image'])) {
            $fileUpdate = (new FileTools())->uploadFile(
                FileTools::$STORE_USER_PROFILE,
                $userUpdate['profile']['image']
            )
                ->store();
        }

        if ($access != User::ACCESS_PERSONAL_ACCOUNT && $access != User::ACCESS_PERSONAL_ACCOUNT_LIMITED) {
            $userOriginal->update(
                array_merge(
                    [
                        'name' => $userUpdate['name'],
                    ],
                    isset($userUpdate['password']) ? [
                        'password' => $userUpdate['password'],
                    ] : []
                )
            );

            if ($user->hasPermissions(['userUpdate']) && CommonTools::anyKeyExists(
                $userUpdate['profile'],
                'address',
                'first_name',
                'last_name',
                'national_code',
                'phone',
                'user_id',
            )) {
                $profile = $userOriginal->profile()
                    ->updateOrCreate(
                        [
                            'user_id' => $userOriginal->id,
                        ],
                        CommonTools::toSafeArray(
                            $userUpdate['profile'],
                            'address',
                            'first_name',
                            'last_name',
                            'national_code',
                            'phone',
                        )
                    );
            }

            $isActive = $user->hasPermissions(['userStatusEdit'])
                && isset($userUpdate['active']) ? $userUpdate['active'] : $userOriginal->active;

            $userOriginal->roles()
                ->sync(
                    [
                        (int) ($userUpdate['role_id']) => [
                            'primary' => $isActive,
                        ],
                    ]

                );
        } else {
            $profile = $userOriginal->profile()
                ->updateOrCreate(
                    [
                        'user_id' => $userOriginal->id,
                    ],
                    CommonTools::toSafeArray(
                        $userUpdate['profile'],
                        'first_name',
                        'last_name',
                        'phone',
                    )
                );
        }

        if (isset($fileUpdate) && isset($userOriginal->profile)) {
            $_profile = $userOriginal->profile ?? $profile;

            if (isset($_profile->image)) {
                $safe_user_image = $_profile->image;

                $_profile->image()
                    ->delete();
            }

            $_profile->image()
                ->create([
                    'name' => $fileUpdate->getFullName(),
                    'extra' => Image::$EXTRA_IMAGE_USER_PROFILE,
                ]);

            if (isset($safe_user_image)) {
                (new FileTools())
                    ->loadFromUserProfile($safe_user_image)
                    ->removeFile();
            }
        }
    }

    public static function deleteUser($userOriginal)
    {
        if ($userOriginal->profile && isset($userOriginal->profile->image)) {
            (new FileTools())
                ->loadFromUserProfile($userOriginal->profile->image)
                ->removeFile();
        }

        $userOriginal->delete();
    }
}
