<?php

namespace App\Http\Controllers;

use App\Enums\ApplicationConfigActions;
use App\Enums\PermissionAccess;
use App\Enums\PermissionWeeklyScheduleAccess;
use App\Http\Requests\WeeklySchedule\DeleteWeeklyScheduleRequest;
use App\Http\Requests\WeeklySchedule\ExcelStoreWeeklyScheduleRequest;
use App\Http\Requests\WeeklySchedule\ExportPDFWeeklyScheduleRequest;
use App\Http\Requests\WeeklySchedule\StoreWeeklyScheduleRequest;
use App\Http\Requests\WeeklySchedule\UpdateWeeklyScheduleRequest;
use App\Http\Requests\WeeklySchedule\ViewWeeklyScheduleRequest;
use App\Imports\WeeklyScheduleImport;
use App\Models\ApplicationConfig;
use App\Models\Bell;
use App\Models\ClassRoom;
use App\Models\Profile;
use App\Models\WeeklySchedule;
use App\Models\User;
use App\Tools\ApplicationConfigTools;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use App\Tools\TermTools;
use DB;
use Excel;
use Maatwebsite\Excel\Validators\ValidationException;
use Morilog\Jalali\Jalalian;
use PDF;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

class WeeklyScheduleController extends Controller
{

    public function index(ViewWeeklyScheduleRequest $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions(PermissionWeeklyScheduleAccess::view);

            $dataTableTools = new DataTableTools();
            $query = WeeklySchedule::with(
                [
                    'bell:id,title,started_at,ended_at',
                    'classRoom:id,major_id,academic_level_id,title,code',
                    'classRoom.major',
                    'classRoom.academicLevel',
                    'lesson',
                    'teacher:id,name',
                    'teacher.profile:id,user_id,first_name,last_name',
                ]
            )
                ->when(
                    $isNotAdmin,
                    fn($qWeeklySchedule) =>
                    $qWeeklySchedule->where('teacher_id', '=', $user->id)
                );

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('weekly_schedules', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'برنامه هفتگی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreWeeklyScheduleRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $weeklyScheduleAdd = CommonTools::safeRequest($request, 'weekly_schedule');

            $teacherId = $weeklyScheduleAdd['teacher_id'];

            $existsTeacher = User::where('id', '=', $teacherId)
                ->whereHas('roles', function ($qRoles) {
                    $qRoles->where('name', '=', PermissionAccess::teacher);
                })
                ->exists();

            if (!$existsTeacher) {
                $profile = Profile::where('user_id', '=', $teacherId)->first();
                $fullName = $profile != null ? "{$profile->first_name} {$profile->last_name}"  : $teacherId;
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessageFormat('messages.user_access_denied', ['user' => $fullName, 'role' => 'معلم'])
                    ->getJsonResponse(Response::HTTP_BAD_REQUEST);
            }

            DB::beginTransaction();

            $weeklySchedule = WeeklySchedule::create($weeklyScheduleAdd);


            $user->appendLog($request, ['new' => $weeklySchedule]);
            DB::commit();

            $weeklySchedule->load(
                [
                    'bell:id,title,started_at,ended_at',
                    'classRoom:id,major_id,academic_level_id,title,code',
                    'classRoom.major',
                    'classRoom.academicLevel',
                    'lesson',
                    'teacher:id,name',
                    'teacher.profile:id,user_id,first_name,last_name',
                ]
            );

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'برنامه هفتگی'])
                ->setData('weekly_schedule',  $weeklySchedule)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'weeklyScheduleStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'برنامه هفتگی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function excelStore(ExcelStoreWeeklyScheduleRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $weeklySchedulesAdd = CommonTools::safeRequest($request, 'xlsx');

            DB::beginTransaction();

            Excel::import(
                new WeeklyScheduleImport(),
                $weeklySchedulesAdd
            );

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_recorded',
                    ['title' => 'برنامه هفتگی']
                )
                ->getJsonResponse();
        } catch (ValidationException $e) {
            DB::rollBack();
            CommonTools::registerException($e, 'weeklyScheduleExcelStore');

            $messageErrors = $e->errors();

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage(array_slice($messageErrors, 0, min(15, count($messageErrors))))
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'weeklyScheduleExcelStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'برنامه هفتگی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }


    public function destroy(DeleteWeeklyScheduleRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $weeklySchedule = WeeklySchedule::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $weeklySchedule->toArray()]);
            $weeklySchedule->delete();


            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'برنامه هفتگی'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'weeklyScheduleDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'برنامه هفتگی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateWeeklyScheduleRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }


        try {
            $user = User::find(auth()->user()->id ?? -1);
            $requestBody = CommonTools::safeRequest($request, 'id', 'weekly_schedule');
            $weeklyScheduleUpdate = $requestBody['weekly_schedule'];
            $weeklyScheduleOriginal = WeeklySchedule::find($requestBody['id']);

            $teacherId = $weeklyScheduleUpdate['teacher_id'];

            $existsTeacher = User::where('id', '=', $teacherId)
                ->whereHas('roles', function ($qRoles) {
                    $qRoles->where('name', '=', PermissionAccess::teacher);
                })
                ->exists();

            if (!$existsTeacher) {
                $profile = Profile::where('user_id', '=', $teacherId)->first();
                $fullName = $profile != null ? "{$profile->first_name} {$profile->last_name}"  : $teacherId;
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessageFormat('messages.user_access_denied', ['user' => $fullName, 'role' => 'معلم'])
                    ->getJsonResponse(Response::HTTP_BAD_REQUEST);
            }

            DB::beginTransaction();

            $user->appendLog($request, [
                'old' => $weeklyScheduleOriginal->toArray(),
                'new' => $weeklyScheduleUpdate,
            ]);
            $weeklyScheduleOriginal->update($weeklyScheduleUpdate);

            DB::commit();

            $weeklyScheduleOriginal->load(
                [
                    'bell:id,title,started_at,ended_at',
                    'classRoom:id,major_id,academic_level_id,title,code',
                    'classRoom.major',
                    'classRoom.academicLevel',
                    'lesson',
                    'teacher:id,name',
                    'teacher.profile:id,user_id,first_name,last_name',
                ]
            );

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'برنامه هفتگی']
                )
                ->setData('weekly_schedule',  $weeklyScheduleOriginal)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();


            CommonTools::registerException($e, 'weeklyScheduleUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'برنامه هفتگی'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function exportPDF(ExportPDFWeeklyScheduleRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::findOrFail(auth()->user()->id);
            $isNotAdmin = !$user->hasPermissions(PermissionWeeklyScheduleAccess::exportPDF);

            $dataTableTools = new DataTableTools();
            $requestBody = CommonTools::safeRequest($request, 'year', 'term');
            $year = $requestBody['year'] ?? Jalalian::now()->getYear();

            $applicationConfigs = ApplicationConfig::where(
                fn($qAppConfig) => $qAppConfig->whereEncrypted(
                    'title',
                    '=',
                    ApplicationConfigActions::schoolControllerInformationUpdate
                )
                    ->orWhereEncrypted(
                        'title',
                        '=',
                        ApplicationConfigActions::schoolControllerPrivateInformationUpdate
                    )
                    ->orWhereEncrypted(
                        'title',
                        '=',
                        ApplicationConfigActions::schoolControllerPrivateTermUpdate
                    )
            )
                ->where(
                    'active',
                    '=',
                    true
                )
                ->get();

            $applicationConfigs = ApplicationConfigTools::mapKey($applicationConfigs);

            $schoolConfig = $applicationConfigs
                ->get(ApplicationConfigActions::schoolControllerInformationUpdate->value)
                ->toArray();
            $schoolPrivateConfig = $applicationConfigs
                ->get(ApplicationConfigActions::schoolControllerPrivateInformationUpdate->value)
                ->toArray();
            $schoolPrivateTerm = $applicationConfigs
                ->get(ApplicationConfigActions::schoolControllerPrivateTermUpdate->value)
                ->toArray();

            $termData = collect($schoolPrivateTerm['value'])->first(fn($el) => $el['term'] == 1);

            if (!$termData) {
                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_ERROR)
                    ->setMessage(__('messages.not_found', ['title' => 'نوبت']))
                    ->getJsonResponse(Response::HTTP_BAD_REQUEST);
            }

            $termDates = TermTools::getDateFromTerm($termData, $year);

            $query = WeeklySchedule::select([
                'id',
                'teacher_id',
                'class_room_id',
                'lesson_id',
                'bell_id',
                'day',
            ])
                ->when(
                    $isNotAdmin,
                    fn($qWeeklySchedule) =>
                    $qWeeklySchedule->where('teacher_id', '=', $user->id)
                )
                ->with([
                    'lesson:id,title',
                    'teacher:id',
                    'teacher.profile:id,user_id,first_name,last_name',
                ])
                ->whereDate(
                    'created_at',
                    '>=',
                    $termDates['started_at'],
                )
                ->whereDate(
                    'created_at',
                    '<',
                    $termDates['started_at']
                        ->addYear(),
                );

            $dataTableTools->doQuery($query, $request)
                ->toArray();

            $weeklySchedules = collect($dataTableTools->data);


            $classRoomIds = $weeklySchedules->pluck('class_room_id')
                ->unique()
                ->values();

            $weeklySchedules = $weeklySchedules->groupBy('class_room_id');

            $classRooms = ClassRoom::select([
                'id',
                'major_id',
                'code',
            ])
                ->with([
                    'major:id,title'
                ])
                ->whereIn(
                    'id',
                    $classRoomIds
                )
                ->get();


            $weeklySchedules = $weeklySchedules->map(
                fn($elWS, $classRoomId) => array_merge(
                    $classRooms->find($classRoomId)->toArray(),
                    [
                        'weekly_schedules' => $elWS->groupBy('day')
                            ->sortKeys()
                            ->map(fn($elWSPerDay) => $elWSPerDay->groupBy('bell_id')
                                ->sortKeys())
                            ->toArray()
                    ]
                )
            )
                ->values();

            $bells = Bell::orderBy('started_at')
                ->get()
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                $pdf = PDF::loadView('pdf.weekly-schedules', [
                    'weeklySchedules' => $weeklySchedules->toArray(),
                    'bells' => $bells,
                    'year' => $year,
                    'reportInfo' => [
                        'school_code' => $schoolPrivateConfig['value']['school_code'],
                        'school_type' => $schoolPrivateConfig['value']['school_type'],
                        'school_title' => $schoolConfig['value']['title'],
                        'school_province' => $schoolConfig['value']['province'],
                        'school_district' => $schoolConfig['value']['district'],
                    ]
                ]);

                return $pdf->stream('document.pdf');
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            CommonTools::registerException($e, 'gradeExportPDF');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_get_information',
                ['title' => 'چاپ نمره نهایی']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
