<?php

namespace App\Http\Requests\Teacher;

use App\Enums\PermissionTeacherAccess;
use App\Http\Requests\Model\ModelBaseRequest;
use App\Models\Teacher;
use App\Rules\NationalCodeRule;
use App\Tools\CommonTools;
use App\Tools\Regex;

class UpdateTeacherRequest extends ModelBaseRequest
{
    protected $authorizePermissions = [
        PermissionTeacherAccess::update
    ];


    public function rules()
    {
        $teacherId = -1;
        $profileId = -1;
        $userId = -1;

        try {
            $teacherId = CommonTools::toSafeHTML($this->get('id'));
            $teacher = Teacher::select(['id', 'user_id'])
                ->with(['user:id', 'user.profile:id,user_id'])
                ->findOrFail(intval($teacherId));
            $userId = $teacher->user_id;
            $profileId = $teacher
                ->user
                ->profile
                ->id;
            //code...
        } catch (\Throwable $th) {
            //throw $th;
        }

        return [
            'id' => [
                'required',
                'int',
                'gt:1',
                'exists:teachers,id'
            ],
            'teacher.personally_code' => [
                'required',
                'string',
                'min:6',
                'max:120',
                "unique_encrypted:teachers,personally_code,$teacherId"
            ],
            'teacher.user.role_id' => [
                'int',
                'exists:roles,id',
                'gt:1'
            ],
            'teacher.user.active' => ['in:0,1,2'],
            'teacher.user.name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::username,
                "unique_encrypted:users,name,$userId"
            ],
            'teacher.user.password' => [
                'confirmed',
                'min:6'
            ],

            'teacher.user.profile' => ['required'],
            'teacher.user.profile.first_name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::alphabeticPersian
            ],
            'teacher.user.profile.last_name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::alphabeticPersian
            ],
            'teacher.user.profile.national_code' => [
                'required',
                'string',
                'max:10',
                new NationalCodeRule(),
                "unique_encrypted:profiles,national_code,$profileId"
            ],
            'teacher.user.profile.address' => [
                'required',
                'string',
                'min:3',
                'max:255',
                'regex:' . Regex::alphabetic
            ],
            'teacher.user.profile.phone' => [
                'required',
                'string',
                'min:3',
                'max:12',
                'regex:' . Regex::englishNumber,
                "unique_encrypted:profiles,phone,$profileId"
            ],
            'teacher.user.profile.image' => [
                'mimes:jpeg,jpg,png,webp,avif',
                'max:1024'
            ]
        ];
    }
}
