<?php

namespace App\Imports;

use App\Enums\PermissionAccess;
use App\Exceptions\AnyException;
use App\Http\Controllers\UserController;
use App\Models\ClassRoom;
use App\Models\Profile;
use App\Rules\NationalCodeRule;
use App\Tools\Regex;
use Arr;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\PersistRelations;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithValidation;

class StudentImport implements ToCollection, WithValidation, WithHeadingRow, WithMapping, PersistRelations
{
    /**
     * @param Collection $collection
     */
    public function collection(Collection $collection)
    {

        $file = null;
        foreach ($collection as $row) {
            $studentAdd = $row['student'];

            $profile = Profile::whereEncrypted(
                'national_code',
                '=',
                Arr::get($studentAdd, 'parent.profile.national_code')
            )
                ->first();

            if ($profile == null) {
                $parentAdd = $studentAdd['parent'];

                if (
                    $parentAdd['profile'] == null ||
                    $parentAdd['profile']['address'] == null ||
                    $parentAdd['profile']['first_name'] == null ||
                    $parentAdd['profile']['last_name'] == null ||
                    $parentAdd['profile']['national_code'] == null ||
                    $parentAdd['profile']['phone'] == null
                ) {
                    throw new AnyException(__('validation.required', [
                        'attribute'
                    ]));
                }

                [$user] = UserController::createUser(
                    $parentAdd,
                    $file,
                    PermissionAccess::parents->value
                );
            } else {
                $user = $profile->user()->firstOrFail();
            }

            $student =  $user->children()
                ->create([
                    'first_name' => $studentAdd['first_name'],
                    'last_name' => $studentAdd['last_name'],
                    'national_code' => $studentAdd['national_code'],
                ]);

            if (
                isset($studentAdd['class_code']) &&
                $studentAdd['class_code'] != null
            ) {
                $classRoom = ClassRoom::where('code', '=', $studentAdd['class_code'])
                    ->firstOrFail();

                $student->classRooms()->attach($classRoom->id);
            }
        }
    }


    public function map($row): array
    {
        try {
            $parentPhone = strval($row['parent_phone']);

            if (strlen($parentPhone) > 0 && $parentPhone[0] != '0') {
                $parentPhone = "0$parentPhone";
            }

            return Arr::undot([
                'student.class_code' => $this->valueOrNull(strval($row['class_code'])),
                'student.first_name' => $row['first_name'],
                'student.last_name' => $row['last_name'],
                'student.national_code' => strval($row['national_code']),
                'student.parent.active' => '1',
                'student.parent.name' =>  $this->valueOrNull('parent_' . $row['parent_national_code']),
                'student.parent.password' =>  $this->valueOrNull(strval($row['parent_national_code'])),
                'student.parent.profile.address' =>  $this->valueOrNull($row['parent_address']),
                'student.parent.profile.first_name' =>  $this->valueOrNull($row['parent_first_name']),
                'student.parent.profile.last_name' =>  $this->valueOrNull($row['parent_last_name']),
                'student.parent.profile.national_code' =>  $this->valueOrNull(strval($row['parent_national_code'])),
                'student.parent.profile.phone' =>  $this->valueOrNull($parentPhone),
            ]);
        } catch (\Throwable $th) {
            //throw $th;
        }

        throw new AnyException(__('messages.error_excel_parse_row'));
    }


    public function rules(): array
    {

        return [
            '*.student.class_code' => [
                'nullable',
                'string',
                'min:1',
                'max:100',
                'exists:class_rooms,code'
            ],
            '*.student.first_name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::alphabeticPersian
            ],
            '*.student.last_name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::alphabeticPersian
            ],
            '*.student.national_code' => [
                'required',
                'string',
                'max:10',
                new NationalCodeRule(),
                "unique_encrypted:profiles,national_code"
            ],
            '*.student.parent.active' => [
                'nullable',
                'in:0,1,2'
            ],
            '*.student.parent.name' => [
                'nullable',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::username
            ],
            '*.student.parent.password' => [
                'nullable',
                'string',
                'min:3',
                'max:20',
            ],
            '*.student.parent.profile.address' => [
                'nullable',
                'string',
                'min:3',
                'max:255',
                'regex:' . Regex::alphabetic
            ],
            '*.student.parent.profile.first_name' => [
                'nullable',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::alphabeticPersian
            ],
            '*.student.parent.profile.last_name' => [
                'nullable',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::alphabeticPersian
            ],
            '*.student.parent.profile.national_code' => [
                'required',
                'string',
                'max:10',
                new NationalCodeRule()
            ],
            '*.student.parent.profile.phone' => [
                'nullable',
                'string',
                'min:3',
                'max:12',
                'regex:' . Regex::englishNumber
            ],
        ];
    }

    private function valueOrNull($value)
    {

        return strlen(strval($value)) > 0 ? $value : null;
    }
}
