<?php

namespace App\Imports;

use App\Exceptions\AnyException;
use App\Models\ClassRoom;
use App\Models\Profile;
use App\Models\WeeklySchedule;
use App\Rules\NationalCodeRule;
use Arr;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\PersistRelations;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithValidation;

class WeeklyScheduleImport implements ToCollection, WithValidation, WithHeadingRow, WithMapping, PersistRelations, SkipsEmptyRows
{
    /**
     * @param Collection $collection
     */
    public function collection(Collection $collection)
    {
        $nationalCodes = $collection->map(
            fn($el) =>
            data_get($el, 'weekly_schedule.teacher_national_code')
        )
            ->unique();

        $classRoomCodes = $collection->map(
            fn($el) =>
            data_get($el, 'weekly_schedule.class_code')
        )
            ->unique();

        $users = Profile::select(['user_id', 'national_code'])
            ->where(
                fn($qProfile) =>
                $nationalCodes->each(
                    fn($elNC) =>
                    $qProfile->orWhereEncrypted('national_code', $elNC)
                )
            )
            ->get()
            ->mapWithKeys(
                fn($el) => [data_get($el, 'national_code') => data_get($el, 'user_id')]
            );

        $classRooms = ClassRoom::select(['id', 'code'])
            ->whereIn(
                'code',
                $classRoomCodes
            )
            ->get()
            ->mapWithKeys(
                fn($el) => [data_get($el, 'code') => data_get($el, 'id')]
            );


        foreach ($collection as $row) {
            $ws = $row['weekly_schedule'];

            WeeklySchedule::create(
                array_merge(
                    Arr::except($ws, ['class_code', 'teacher_national_code']),
                    [
                        'class_room_id' => $classRooms->get($ws['class_code']),
                        'teacher_id' => $users->get($ws['teacher_national_code']),
                        'day' => intval($ws['day']) - 1,
                        'bell_type' => $ws['bell_type'] === 'عادی' ?
                            'normal' : ($ws['bell_type'] === 'زوج' ? 'even' : 'odd')
                    ]
                )
            );
        }
    }


    public function map($row): array
    {
        try {
            return  [
                'weekly_schedule' => [
                    'day' => strval($row['day']),
                    'class_code' => strval($row['class_code']),
                    'lesson_id' => strval($row['lesson_id']),
                    'teacher_national_code' => strval($row['teacher_national_code']),
                    'bell_id' => strval($row['bell_id']),
                    'bell_type' => strval($row['bell_type']),
                ]
            ];
        } catch (\Throwable $th) {
            //throw $th;
        }

        throw new AnyException(__('messages.error_excel_parse_row'));
    }

    public function isEmptyWhen(array $row): bool
    {
        return !isset($row['day']) &&
            !isset($row['class_code']) &&
            !isset($row['lesson_id']) &&
            !isset($row['teacher_national_code']) &&
            !isset($row['bell_id']) &&
            !isset($row['bell_type']);
    }

    public function rules(): array
    {
        return [
            '*.weekly_schedule.day' => [
                'required',
                'numeric',
                'min:1',
                'max:7',
            ],
            '*.weekly_schedule.class_code' => [
                'required',
                'string',
                'min:1',
                'max:50',
                'exists:class_rooms,code',
            ],
            '*.weekly_schedule.lesson_id' => [
                'required',
                'numeric',
                'exists:lessons,id',
            ],
            '*.weekly_schedule.teacher_national_code' => [
                'required',
                'string',
                'max:10',
                new NationalCodeRule(),
                "exists_encrypted:profiles,national_code"
            ],
            '*.weekly_schedule.bell_id' => [
                'required',
                'numeric',
                'exists:bells,id',
            ],
            '*.weekly_schedule.bell_type' => [
                'required',
                'string',
                'in:عادی,زوج,فرد',
            ],
        ];
    }
}
