<?php

namespace App\Jobs;

use App\Models\WeeklySchedule;
use App\Tools\CommonTools;
use Carbon\Carbon;
use DB;
use Morilog\Jalali\Jalalian;

class TeacherPresenceJob
{
    const MINIMUM_TIME = '00:15:00'; //15 minutes

    /**
     * Create a new job instance.
     */
    public function __construct() {}

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $nowYMD = Carbon::now()->format('Y-m-d');
            $day = Jalalian::now()->getDayOfWeek();

            $weeklySchedules = WeeklySchedule::with([
                'bell:id,started_at,ended_at'
            ])
                ->whereHas(
                    'bell',
                    fn($qBell) =>
                    $qBell
                        ->whereRaw(
                            //add time 03:30+ for timezone iran
                            "SUBTIME(`bells`.`started_at`, '03:30:00') <= SUBTIME(CURTIME(), ?)",
                            [self::MINIMUM_TIME]
                        )
                )
                ->where('day', '=', $day)
                ->where(
                    fn($qWeeklySchedules) =>
                    $qWeeklySchedules
                        ->whereDoesntHave('schoolSessions')
                        ->orWhereDoesntHave(
                            'schoolSessions',
                            fn($qSchoolSessions) =>
                            $qSchoolSessions
                                ->join('bells', 'bells.id', '=', 'weekly_schedules.bell_id')
                                ->whereRaw(
                                    //add time 03:30+ for timezone iran
                                    "TIME(school_sessions.created_at) BETWEEN 
                                            SUBTIME(bells.started_at, '03:35:00') 
                                            AND 
                                            ADDTIME(bells.ended_at, '03:35:00')"

                                )
                                ->whereToday('school_sessions.created_at')
                        )
                )
                ->get();


            if ($weeklySchedules->count() <= 0) {
                return;
            }

            $weeklySchedules->chunk(100)
                ->each(
                    fn($chunkWeeklySchedules) =>
                    DB::transaction(function () use ($nowYMD, $chunkWeeklySchedules) {
                        $chunkWeeklySchedules->each(
                            fn($elWS) =>
                            $elWS
                                ->schoolSessions()
                                ->create([
                                    'teacher_user_id' => $elWS->teacher_id,
                                    'created_at' => Carbon::createFromFormat(
                                        'Y-m-d H:i:s',
                                        sprintf(
                                            '%s %s',
                                            $nowYMD,
                                            $elWS->bell->started_at
                                        ),
                                        'Asia/Tehran'
                                    )
                                        ->setTimezone('UTC'),
                                    'held' => false,
                                ])
                        );
                    })
                );
        } catch (\Throwable $th) {
            CommonTools::registerException($th, 'TeacherPresenceJob');
        }
    }
}
