<?php

namespace App\Rules;

use App\Models\Student;
use App\Models\WeeklySchedule;
use Arr;
use Carbon\Carbon;
use Closure;
use Illuminate\Contracts\Validation\DataAwareRule;
use Illuminate\Contracts\Validation\ValidationRule;
use Morilog\Jalali\Jalalian;

class OwnStudentWithSchoolSessionRule implements DataAwareRule, ValidationRule
{
    private $data;
    private $schoolSessionField;
    private $studentField;

    /**
     * validation own students with school_session_id
     * @param string $studentField
     * @param string $schoolSessionField
     */
    public function __construct($studentField, $schoolSessionField)
    {
        $this->studentField = $studentField;
        $this->schoolSessionField = $schoolSessionField;
    }

    /**
     * Run the validation rule.
     *
     * @param  \Closure(string, ?string=): \Illuminate\Translation\PotentiallyTranslatedString  $fail
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        $tmpSchoolSessionField = $this->schoolSessionField;
        if ($groupIdx = strpos($tmpSchoolSessionField, '*')) {
            $tmpSchoolSessionField = str_replace(
                substr($tmpSchoolSessionField, 0, $groupIdx + 2),
                str_replace($this->studentField, '', $attribute),
                $tmpSchoolSessionField
            );
        }

        $schoolSessionId =  Arr::get($this->data, $tmpSchoolSessionField);

        $realStudentIds = Student::whereHas(
            'classRooms',
            fn($qClassRooms) =>
            $qClassRooms
                ->whereHas(
                    'weeklySchedules',
                    fn($qWeeklySchedules) =>
                    $qWeeklySchedules
                        ->whereHas(
                            'schoolSessions',
                            fn($qSchoolSessions) =>
                            $qSchoolSessions->where('school_sessions.id', '=', $schoolSessionId)
                        )
                )
        )
            ->where('students.id',  $value)
            ->pluck('id');

        $notAccessStudentIds = array_values(array_diff([$value], $realStudentIds->toArray()));

        if (count($notAccessStudentIds) == 0) {
            return;
        }

        $notAccessStudents = Student::select(['id', 'first_name', 'last_name'])
            ->whereIn('id', $notAccessStudentIds)
            ->get();

        $weeklySchedule = WeeklySchedule::whereHas(
            'schoolSessions',
            fn($qSchoolSessions) =>
            $qSchoolSessions->where(
                'school_sessions.id',
                '=',
                $schoolSessionId
            )
        )
            ->first();

        $dataTime = '-';
        $teacherFullName = '-';

        if ($weeklySchedule != null) {
            if ($teacher = $weeklySchedule->teacher()->first()) {
                if ($teacherProfile = $teacher->profile()->first()) {
                    $teacherFullName = $teacherProfile->first_name . ' ' . $teacherProfile->last_name;
                }
            }
            $dataTime = Jalalian::fromCarbon(
                $weeklySchedule->created_at instanceof Carbon ?
                    $weeklySchedule->created_at :
                    Carbon::createFromTimeString($weeklySchedule->created_at)
            )
                ->format('Y-m-d H:i:s');
        }


        $fail('validation.own_student_with_school_session', null)
            ->translate([
                'attribute' => __(
                    'validation.attributes',
                )[preg_replace('/\.\d+$/', '.*', $attribute)] ?? $attribute,
                'students' => $notAccessStudents
                    ->map(
                        fn($v) =>
                        $v['first_name'] . ' ' . $v['last_name']
                    )
                    ->join('، '),
                'dataTime' => $dataTime,
                'teacherFullName' => $teacherFullName
            ]);
    }


    public function setData(array $data)
    {
        $this->data = $data;
    }
}
