<?php

namespace App\Rules;

use Closure;
use Illuminate\Contracts\Validation\ValidationRule;
use Illuminate\Support\Facades\DB;

class PolymorphicExistsRule implements ValidationRule
{

    protected $table;
    protected $morph;
    protected $model;
    protected $ignoreId;

    /**
     * Create a new rule instance.
     *
     * @param string $table (images)
     * @param string $morph (related(_type|_id)) imageable -> imageable_type, imageable_id
     * @param string $model App\Models\User
     * @param mixed [$ignoreId] id
     */
    public function __construct(string $table, string $morph, string $model, mixed $ignoreId = null)
    {
        $this->table = $table;
        $this->morph = $morph;
        $this->model = $model;
        $this->ignoreId = $ignoreId;
    }

    /**
     * Run the validation rule.
     *
     * @param  string  $attribute
     * @param  mixed  $value
     * @param  Closure  $fail
     * @return void
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        $exists = DB::table($this->table)
            ->where('id', $value)
            ->where($this->morph . '_type', '=', $this->model)
            ->when($this->ignoreId, function ($q, $ignoreId) {
                $q->where($this->morph . '_id', '!=', $ignoreId);
            })
            ->exists();

        if (!$exists) {
            $fail('validation.polymorphic_exists')
                ->translate([
                    'attribute' => __(
                        'validation.attributes',
                    )[preg_replace('/\.\d+$/', '.*', $attribute)] ?? $attribute,
                    'value' => $value,
                    'morph' => $this->morph,
                    'model' => trans(
                        'validation.attributes.' . $this->model,
                        [],
                        $this->model
                    ),
                ]);
        }
    }
}
