<?php

namespace App\Tools;

use ErrorException;
use Exception;
use Illuminate\Encryption\Encrypter;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class FileTools
{
    private const CIPHER = 'AES-256-CBC';
    private const HASH = 'md5';

    static $TYPE_IMAGE = ['png', 'jpg', 'jpeg', 'gif', 'webp', 'avif'];
    static $TYPE_VOICE = ['acc', 'mp3'];
    static $TYPE_VIDEO = ['mp4', 'mkv'];
    static $TYPE_PDF = ['pdf'];
    static $TYPE_TEXT = ['txt'];
    static $IMAGE = "images";
    static $VOICE = "voices";
    static $VIDEO = "videos";
    static $PDF = "pdfs";
    static $TEXT = "texts";

    static $STORE_APP_LOG = 'app\log';
    static $STORE_USER = 'users';
    static $STORE_STUDENT = 'students';
    static $STORE_USER_PROFILE = 'users\profiles';
    static $STORE_SCHOOL = 'schools';

    /**
     * @var string
     */
    protected $key = '';

    /**
     * @var resource|string
     */
    protected $content;

    /**
     * @var string
     */
    protected $fileName;

    /**
     * @var string
     */
    protected $extension;

    /**
     * @var string
     */
    protected $type;

    /**
     * @var string
     */
    protected $extraPath;

    /**
     * @var string
     */
    protected $storeAs;

    /**
     * @var UploadedFile|UploadedFile[]
     */
    protected $file;

    /**
     * file is private
     * @var bool
     */
    protected bool $isPrivate = false;

    protected $isCache = false;

    public function __construct($isCache = false)
    {
        $this->isCache = $isCache;
    }

    public function publicDir()
    {
        $this->isPrivate = false;
        return $this;
    }

    public function privateDir()
    {
        $this->isPrivate = true;
        return $this;
    }

    public function setIsCache($isCache = false)
    {
        $this->isCache = $isCache;
    }

    public static function getExtensionFromStr($filename)
    {
        $index = strpos($filename, ".");

        if ($index != false) {
            return substr($filename, $index + 1);
        }

        $index = strpos($filename, "/");

        if ($index != false) {
            return substr($filename, $index + 1);
        }

        return null;
    }

    public static function splitFilenameAndExtension($filename)
    {
        $extension = FileTools::getExtensionFromStr($filename);
        return [
            str_replace(
                '.' . $extension,
                '',
                $filename
            ),
            $extension,
        ];
    }

    public function loadFile($file, $storeAs)
    {
        $splitFileName = FileTools::splitFilenameAndExtension(isset($file['name']) ? ($file->name ?? $file['name']) : $file);
        $this->fileName = $splitFileName[0];
        $this->extension = $splitFileName[1];
        $this->storeAs = $storeAs;

        $this->setType();

        return $this;
    }

    public function loadContent($content, $fileName, $extension)
    {
        $this->fileName = $fileName;
        $this->extension = $extension;
        $this->content = $content;

        $this->setType();

        return $this;
    }

    public function loadContentFile()
    {
        $this->content = Storage::get(
            $this->getPathFile()
        );

        return $this;
    }

    public function loadFromAppLog($logFile)
    {
        return $this->loadFile($logFile, FileTools::$STORE_APP_LOG);
    }


    public function loadFromUser($userFile)
    {
        return $this->loadFile($userFile, FileTools::$STORE_USER);
    }

    public function loadFromStudent($studentFile)
    {
        return $this->loadFile($studentFile, FileTools::$STORE_STUDENT);
    }

    public function loadFromUserProfile($userProfileFile)
    {
        return $this->loadFile($userProfileFile, FileTools::$STORE_USER_PROFILE);
    }

    private function setType()
    {
        if (in_array($this->extension, self::$TYPE_IMAGE)) {
            $this->type = self::$IMAGE;
        } elseif (in_array($this->extension, self::$TYPE_VOICE)) {
            $this->type = self::$VOICE;
        } elseif (in_array($this->extension, self::$TYPE_VIDEO)) {
            $this->type = self::$VIDEO;
        } elseif (in_array($this->extension, self::$TYPE_PDF)) {
            $this->type = self::$PDF;
        } elseif (in_array($this->extension, self::$TYPE_TEXT)) {
            $this->type = self::$TEXT;
        } else {
            throw new ErrorException("invalid type");
        }
    }

    public function uploadFile($storeAs, $file, $filename = null)
    {
        $this->file = $file;
        $this->extension = $this->file->getClientOriginalExtension();
        $this->storeAs = $storeAs;

        $this->setType();
        $this->fileName = $filename ?? RandomGenerator::filename($this->getDirFile());

        return $this;
    }

    public function uploadEncryptFile($storeAs, $file)
    {
        $this->uploadFile($storeAs, $file);

        $this->content = $file->getContent();

        return $this;
    }

    public function cloneFileTools(
        $key = null,
        $content = null,
        $fileName = null,
        $extension = null,
        $type = null,
        $extraPath = null,
        $storeAs = null,
        $file = null
    ) {
        $this->key = $key;
        $this->content = $content;
        $this->fileName = $fileName;
        $this->extension = $extension;
        $this->type = $type;
        $this->extraPath = $extraPath;
        $this->storeAs = $storeAs;
        $this->file = $file;

        return $this;
    }

    /**
     * get absolute dir of file
     * @param mixed $ignoreCache
     * @return string "temporary_media/videos"
     */
    public function getDirFile($ignoreCache = false)
    {
        if ($ignoreCache === false && $this->isCache) {
            return sprintf(
                'temp/%s/%s%s',
                $this->storeAs,
                isset($this->extraPath) ? ($this->extraPath . '/') : '',
                $this->type,
            );
        }

        return sprintf(
            '%s/%s%s',
            $this->storeAs,
            isset($this->extraPath) ? ($this->extraPath . '/') : '',
            $this->type,
        );
    }

    public function store()
    {
        if ($this->isPrivate) {
            if (
                $this->file->storeAs($this->getDirFile(true), $this->getFullName()) === false
            ) {
                throw new ErrorException('save File private');
            }
            return $this;
        }

        if (
            $this->file->storePubliclyAs(
                $this->getDirFile(true),
                $this->getFullName(),
                "public"
            ) === false
        ) {
            throw new ErrorException('save File public');
        }

        return $this;
    }

    public function moveToMainFromCacheFolder()
    {
        Storage::move($this->getPathFile(), $this->getPathFile(true));
        return $this;
    }

    public function storeAsCache()
    {
        if (
            $this->file->storeAs(
                $this->getDirFile(true),
                $this->getFullName(),
                ['disk' => 'temp']
            ) === false
        ) {
            throw new ErrorException('save File');
        }

        $cloneFileTools = clone $this;

        $cloneFileTools->setIsCache(true);

        return $cloneFileTools;
    }

    /**
     * encrypt file
     * @param string|null $md5Key
     * @return FileTools
     */
    public function encrypt(string $md5Key = null): FileTools
    {
        $encrypt = new Encrypter($md5Key ?? $this->key, self::CIPHER);

        $this->content = $encrypt->encrypt($this->content);

        return $this;
    }

    /**
     * encrypt file
     * @param string|null $md5Key
     * @return FileTools
     */
    public function encryptDownload(string $md5Key): FileTools
    {
        $salt = openssl_random_pseudo_bytes(256);
        $iv = openssl_random_pseudo_bytes(16);
        $iterations = 999;
        $key = hash_pbkdf2("sha512", $md5Key, $salt, $iterations, 64);

        $encrypted_data = openssl_encrypt($this->content, 'aes-256-cbc', hex2bin($key), OPENSSL_RAW_DATA, $iv);

        $data = array("value" => base64_encode($encrypted_data), "iv" => bin2hex($iv), "salt" => bin2hex($salt));

        $this->content = base64_encode(json_encode($data));

        return $this;
    }

    /**
     * decrypt file
     * @param string|null $key
     * @return FileTools
     */
    public function decrypt(string $key = null): FileTools
    {
        $encrypt = new Encrypter($key ?? $this->key, self::CIPHER);

        $this->content = $encrypt->decrypt($this->content);

        return $this;
    }

    /**
     * store encrypt file
     * @throws ErrorException
     */
    public function encryptStore()
    {
        if (
            Storage::put(
                $this->getPathFile(true),
                $this->content
            ) === false
        ) {
            throw new ErrorException('save encrypt File');
        }

        return $this;
    }

    /**
     * store encrypt file
     * @throws ErrorException
     */
    public function encryptStoreAsCache()
    {
        if (
            Storage::put(
                ($this->isCache ? '' : 'temp/') . $this->getPathFile(),
                $this->content,
                ['disk' => 'temp']

            ) === false
        ) {
            throw new ErrorException('save File');
        }

        $cloneFileTools = clone $this;

        $cloneFileTools->setIsCache(true);

        return $cloneFileTools;
    }

    public function getContent()
    {
        return $this->content;
    }

    /**
     * 
     * @return string 182b0166e59bb7ebdefa1dca1810e7d5.jpg
     */
    public function getFullName()
    {
        return sprintf(
            '%s.%s',
            $this->fileName,
            $this->extension
        );
    }

    /**
     * @return string test.jpg
     */
    public function getOriginalName()
    {
        return $this->file->getClientOriginalName();
    }

    /**
     * get absolute dir and filename of file
     * @param mixed $ignoreCache
     * @return string temporary_media/videos/5d042d2140a6e73195508f2f4b631101.mp4
     */
    public function getPathFile($ignoreCache = false)
    {
        return sprintf(
            '%s/%s',
            $this->getDirFile($ignoreCache),
            $this->getFullName()
        );
    }

    /**
     * 
     * @return string .../{{PROJECT_NAME}}/storage/app/temp/public/users/profiles/images/182b0166e59bb7ebdefa1dca1810e7d5.jpg
     */
    public function getStoragePathFile()
    {
        return Storage::disk($this->getDisk())
            ->path($this->getPathFile());
    }

    /**
     * get storage disk
     * @return string local or public
     */
    public function getDisk()
    {
        return $this->isPrivate ? 'local' : 'public';
    }

    public function removeFile()
    {
        try {
            Storage::disk($this->getDisk())
                ->delete($this->getPathFile());
        } catch (Exception $exception) {
            Log::error($exception->getMessage());
        }

        return $this;
    }

    /**
     * decrypt file
     * 
     * unset content
     * 
     * @return FileTools
     */
    public function removeTemp(): FileTools
    {
        unset($this->content);
        return $this;
    }

    public function setExtraPath($path)
    {
        $this->extraPath = $path;
        return $this;
    }

    public function setKey($key)
    {
        $this->key = hash(self::HASH, $key);
        return $this;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @return string 182b0166e59bb7ebdefa1dca1810e7d5
     */
    public function getFileName(): string
    {
        return $this->fileName;
    }

    /**
     * @return string jpg|png|mp4|...
     */
    public function getExtension(): string
    {
        return $this->extension;
    }

    public function exists($ignoreCache = false)
    {
        return Storage::disk($this->getDisk())
            ->exists($this->getDirFile($ignoreCache) . '/' . $this->getFullName());
    }


    /**
     * move file to path !Important! private/file.jpg to private/test.jpg
     * @param mixed $to
     * @param mixed $ignoreCache
     * @return bool
     */
    public function move($to, $ignoreCache = false)
    {
        return Storage::disk($this->getDisk())
            ->move($this->getDirFile($ignoreCache) . '/' . $this->getFullName(), $to);
    }

    /**
     * copy file to path !Important! private/file.jpg to public/test.jpg
     * @param mixed $disk local, public
     * @param mixed $ignoreCache
     * @return array [bool, FileTools]
     */
    public function diskCopy($disk, $storeAs = null, $ignoreCache = false)
    {
        $originalFile = Storage::disk($this->getDisk())
            ->get($this->getPathFile($ignoreCache));

        $tmpFileTool =  clone $this;

        if ($storeAs != null) {
            $tmpFileTool->storeAs = $storeAs;
        }

        $result =  Storage::disk($disk)
            ->put(
                $tmpFileTool->getPathFile($ignoreCache),
                $originalFile
            );

        return [$result, $tmpFileTool];
    }

    /**
     * rename file to $filename like test.mp4 -> tmp.mp4
     * @param mixed $filename
     * @param mixed $ignoreCache
     * @return bool
     */
    public function rename($filename, $ignoreCache = false)
    {
        $result =  $this->move($this->getDirFile($ignoreCache) . '/' . $filename, $ignoreCache);

        if ($result == true) {
            $splitFileName = FileTools::splitFilenameAndExtension($filename);
            $this->fileName = $splitFileName[0];
            $this->extension = $splitFileName[1];
            $this->setType();
        }

        return $result;
    }

    static function getFileMimeType(string $filePath): string
    {
        if (!file_exists($filePath)) {
            throw new ErrorException("getFileMimeType: invalid type");
        }

        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $filePath);
        finfo_close($finfo);

        return $mimeType;
    }

    static function getFileCategoryType(string $filePath)
    {
        $mimeType = self::getFileMimeType($filePath);

        if (str_starts_with($mimeType, 'image/')) {
            return 'image';
        } elseif (str_starts_with($mimeType, 'video/')) {
            return 'video';
        } elseif (str_starts_with($mimeType, 'text/') || in_array($mimeType, [
            'application/json',
            'application/xml',
            'application/x-yaml',
        ])) {
            return 'text';
        } elseif (str_starts_with($mimeType, 'audio/')) {
            return 'audio';
        } elseif ($mimeType === 'application/pdf') {
            return 'pdf';
        }

        throw new ErrorException("getFileCategoryType: invalid type");
    }
}
